/*
 * Copyright © 2020 Cask Data, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package io.cdap.delta.api;

import java.util.Objects;

/**
 * Represents properties of source in replicator pipeline that are exposed to the targets.
 */
public class SourceProperties {

  /**
   * Enum describing ordering of events.
   */
  public enum Ordering {
    ORDERED,
    UN_ORDERED
  }

  private final boolean rowIdSupported;
  private final Ordering ordering;

  private SourceProperties(boolean rowIdSupported) {
    this(rowIdSupported, Ordering.ORDERED);
  }

  private SourceProperties(boolean rowIdSupported, Ordering ordering) {
    this.rowIdSupported = rowIdSupported;
    this.ordering = ordering;
  }

  public Ordering getOrdering() {
    return ordering;
  }
  public boolean isRowIdSupported() {
    return rowIdSupported;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }
    SourceProperties that = (SourceProperties) o;
    return rowIdSupported == that.rowIdSupported && ordering == that.ordering;
  }

  @Override
  public int hashCode() {
    return Objects.hash(rowIdSupported, ordering);
  }

  /**
   * Builder for a source properties.
   */
  public static class Builder {
    private boolean rowIdSupported;
    private Ordering ordering;

    public Builder() {
      this.ordering = Ordering.ORDERED;
    }


    /**
     * Sets whether the events generated by source have a row id to uniquely identify a row.
     *
     * @return this builder
     */
    public SourceProperties.Builder setRowIdSupported(boolean rowIdSupported) {
      this.rowIdSupported = rowIdSupported;
      return this;
    }

    /**
     * Sets whether the events generated by source are ordered.
     * the order of {@link Ordering#UN_ORDERED unordered} events should be identified by its {
     * @link ChangeEvent#getSourceTimestampMillis()}
     * source time
     * stamp}
     * @return this builder
     */
    public SourceProperties.Builder setOrdering(Ordering ordering) {
      this.ordering = ordering;
      return this;
    }

    /**
     * @return an instance of {@code SourceProperties}
     */
    public SourceProperties build() throws IllegalArgumentException {
      return new SourceProperties(rowIdSupported, ordering);
    }
  }
}
