package io.cana.steam.online.container;

import io.cana.steam.common.bean.Peer;
import io.cana.steam.online.constant.OnlineConstant;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

/**
 * @author QuDasheng
 * @create 2020-06-16 15:24
 */
@Slf4j
@Getter
public class OnlineWheel {
    /**
     * wheel peer index
     */
    private final int index;

    /**
     * wheel peer list
     */
    private final long[] peers;

    /**
     * peer address
     */
    private final long now;

    /**
     * all peer list used length
     */
    private int used;

    public OnlineWheel(long nowPeer) {
        this(0, nowPeer);
    }

    public OnlineWheel(int nowIndex, long nowPeer) {
        this.now = nowPeer;
        this.index = nowIndex;
        this.peers = new long[OnlineConstant.Default.WHEEL_SIZE];
        this.peers[nowIndex] = nowPeer;
        this.used = 1;
    }

    public OnlineWheel(int nowIndex, long nowPeer, long[] snapPeers) {
        this.now = nowPeer;
        this.index = nowIndex;
        final int size = snapPeers.length;
        final long[] peers = new long[size];
        for (int i = 0; i < size; i++) {
            final long snapPeer = snapPeers[i];
            if (snapPeer > 0) {
                peers[i] = snapPeer;
                this.used += 1;
            }
        }
        this.peers = peers;
        this.peers[nowIndex] = nowPeer;
        this.used += 1;
    }

    public long getPeer(int idx) {
        if (idx < 0) {
            return 0;
        }
        return peers[idx];
    }

    public boolean addPeer(int idx, long peer) {
        if (idx >= peers.length) {
            return false;
        }

        peers[idx] = peer;
        used += 1;
        return true;
    }

    public long delPeer(int idx) {
        if (idx >= 0) {
            long peer = peers[idx];
            if (peer > 0) {
                peers[idx] = 0;
                used -= 1;
                return peer;
            }
        }
        return 0;
    }

    public int delPeer(long peer) {
        int idx = indexOf(peer);
        if (idx >= 0) {
            peers[idx] = 0;
            used -= 1;
            return idx;
        }
        return -1;
    }

    public int delPeer(Peer peer) {
        return delPeer(peer.toNum());
    }

    public long nextPeer(long inPeer) {
        int idx = -1;
        long first = -1;
        for (int i = 0; i < peers.length; i++) {
            long peer = peers[i];
            if (peer > 0) {
                if (idx >= 0) {
                    return peer;
                }
                if (first < 0) {
                    first = peer;
                }
                if (peer == inPeer) {
                    idx = i;
                }
            }
        }
        return first;
    }

    public long nextPeer(Peer inPeer) {
        return nextPeer(inPeer.toNum());
    }

    public long nextPeer() {
        return nextPeer(now);
    }

    public int indexOf(long peer) {
        if (peer <= 0) {
            return -1;
        }
        for (int i = 0; i < peers.length; i++) {
            if (peers[i] == peer) {
                return i;
            }
        }
        return -1;
    }

    public int indexOf(Peer peer) {
        return indexOf(peer.toNum());
    }

    public boolean isFull() {
        return used >= peers.length;
    }

    public int getSize() {
        return peers.length;
    }
}
