package io.cana.steam.online.container;

import io.cana.steam.common.utils.CollectionUtil;
import io.cana.steam.online.Member;
import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * @author QuDasheng
 * @create 2020-06-22 10:25
 */
@Slf4j
public class OnlineDrawer {
    /**
     * all peer list
     */
    private final HashMap<Long, Member> members;

    /**
     * all peer by group
     */
    private final HashMap<String, Set<Member>> groups;


    public OnlineDrawer() {
        this.members = CollectionUtil.newHashMap(64);
        this.groups = CollectionUtil.newHashMap(8);
    }

    private OnlineDrawer(HashMap<Long, Member> members, HashMap<String, Set<Member>> groups) {
        this.members = members;
        this.groups = groups;
    }

    public void addMember(Member member) {
        if (null == member || null == members || null == groups) {
            return;
        }

        Long peer = member.getPeer();
        members.put(peer, member);

        String group = member.getGroup();
        Set<Member> groupList = groups.getOrDefault(group, new HashSet<>());
        if (!groupList.contains(member)) {
            synchronized (OnlineDrawer.class) {
                if (!groupList.contains(member)) {
                    groupList.add(member);
                    groups.put(group, groupList);
                }
            }
        }
    }

    public void updateMember(Member member) {
        if (null == member || null == members || null == groups) {
            return;
        }

        Long peer = member.getPeer();
        members.put(peer, member);
    }

    public Member removeMember(Long peer) {
        if (null == members || null == groups) {
            return null;
        }
        Member member = members.remove(peer);
        if (null == member) {
            return null;
        }

        Set<Member> groupList = groups.get(member.getGroup());
        if (CollectionUtil.isEmpty(groupList)) {
            return member;
        }
        groupList.remove(member);
        return member;
    }

    public Member getMember(Long peer) {
        return members.get(peer);
    }

    public int allSize() {
        return members.size();
    }

    public HashMap<Long, Member> allMembers() {
        return members;
    }

    public int groupSize(String group) {
        Set<Member> set = groups.get(group);
        return null == set ? 0 : set.size();
    }

    public List<Member> groupMembers(String group) {
        Set<Member> set = groups.get(group);
        return null == set ? new ArrayList<>() : new ArrayList<>(set);
    }

    public static OnlineDrawer fromMembers(Collection<Member> list) {
        if (CollectionUtil.isEmpty(list)) {
            return new OnlineDrawer();
        }

        HashMap<Long, Member> members = CollectionUtil.newHashMap(list.size());
        HashMap<String, Set<Member>> groups = new HashMap<>(16);
        for (Member member : list) {
            if (null != member) {
                Long peer = member.getPeer();
                String group = member.getGroup();

                members.put(peer, member);
                Set<Member> groupList = groups.getOrDefault(group, new HashSet<>());
                groupList.add(member);
                groups.put(group, groupList);
            }
        }
        return new OnlineDrawer(members, groups);
    }
}
