package io.cana.steam.online;


import io.cana.steam.common.bean.Peer;
import io.cana.steam.common.kind.ClassLoader;
import io.cana.steam.online.container.OnlineDrawer;
import io.cana.steam.online.container.OnlineState;
import io.cana.steam.online.exception.ConstructFailedException;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * @author Dasheng
 * @create 2020-06-16 15:03
 */
@Slf4j
public abstract class Online {
    /**
     * drawer
     */
    protected OnlineDrawer drawer = new OnlineDrawer();

    /**
     * start online
     *
     * @param local
     * @param seeds
     */
    public abstract boolean start(Member local, Peer... seeds);

    /**
     * leave online
     */
    public abstract boolean leave();

    /**
     * peer state
     */
    public abstract OnlineState state();

    /**
     * get drawer
     */
    public OnlineDrawer getDrawer() {
        return drawer;
    }

    /**
     * get member by peer
     */
    public Member getMember(Long peer) {
        return Objects.requireNonNull(this.drawer).getMember(peer);
    }

    /**
     * update member
     */
    public void updateMember(Member member) {
        Objects.requireNonNull(this.drawer).updateMember(member);
    }

    /**
     * remove member
     */
    public void removeMember(Long peer) {
        Objects.requireNonNull(this.drawer).removeMember(peer);
    }

    /**
     * get all members
     */
    public List<Member> allMembers() {
        return new ArrayList<>(Objects.requireNonNull(this.drawer).allMembers().values());
    }

    /**
     * get all peer in group
     */
    public List<Member> groupMembers(String group) {
        return new ArrayList<>(Objects.requireNonNull(this.drawer).groupMembers(group));
    }

    /**
     * bootstrap
     */
    public static Online bootstrap(Member local, Peer... seeds) {
        try {
            final Online online = implement(Online.class);
            if (online.start(local, seeds)) {
                return online;
            } else {
                return null;
            }
        } catch (Exception e) {
            log.error("[CELLA] Create online instance error", e);
            throw new RuntimeException(e);
        }
    }

    /**
     * implement
     */
    private static <T> T implement(Class<T> clazz, Object... params) {
        try {
            return ClassLoader.constructImplementObject("META-INF/online/" + clazz.getName(), params);
        } catch (ClassNotFoundException | IllegalAccessException | InstantiationException | NoSuchMethodException | InvocationTargetException e) {
            throw new ConstructFailedException(e.getMessage(), e);
        }
    }
}
