package io.cana.steam.online;

import io.cana.steam.common.bean.Peer;
import io.cana.steam.online.constant.OnlineConstant.Default;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.ToString;

import java.util.HashMap;
import java.util.Map;

/**
 * @author Dasheng
 * @create 2020-06-16 15:28
 */
@Getter
@EqualsAndHashCode(of = {"peer"})
@ToString
public class Member {
    /**
     * peer key
     *
     * @see Peer#toNum
     */
    private final long peer;

    /**
     * peer group
     */
    private final String group;

    /**
     * peer metadata
     */
    private final Map<String, String> metadata;

    public static Member fromPort(int port) {
        return new Member(new Peer(port).toNum(), Default.DEFAULT_GROUP, new HashMap<>());
    }

    public static Member fromPort(int port, String group) {
        return new Member(new Peer(port).toNum(), group, new HashMap<>());
    }

    public static Member fromPort(int port, Map<String, String> metadata) {
        return new Member(new Peer(port).toNum(), Default.DEFAULT_GROUP, metadata);
    }

    public static Member fromPort(int port, String group, Map<String, String> metadata) {
        return new Member(new Peer(port).toNum(), group, metadata);
    }

    public static Member fromPeer(long peer) {
        return new Member(peer, Default.DEFAULT_GROUP, new HashMap<>());
    }

    public static Member fromPeer(long peer, String group) {
        return new Member(peer, group, new HashMap<>());
    }

    public static Member fromPeer(long peer, Map<String, String> metadata) {
        return new Member(peer, Default.DEFAULT_GROUP, metadata);
    }

    public static Member fromPeer(long peer, String group, Map<String, String> metadata) {
        return new Member(peer, group, metadata);
    }

    private Member(long peer, String group, Map<String, String> metadata) {
        if (peer < 0) {
            this.peer = new Peer(-1).toNum();
        } else {
            this.peer = peer;
        }
        this.group = group;
        this.metadata = metadata;
    }

    public Object put(String key, int value) {
        return metadata.put(key, String.valueOf(value));
    }

    public Object put(String key, long value) {
        return metadata.put(key, String.valueOf(value));
    }

    public Object put(String key, boolean value) {
        return metadata.put(key, String.valueOf(value));
    }

    public Object put(String key, String value) {
        return metadata.put(key, value);
    }

    public void put(Map<String, String> map) {
        metadata.putAll(map);
    }

    public int get(String key, int def) {
        final String value = metadata.get(key);
        if (null == value) {
            return def;
        }
        return Integer.parseInt(value);
    }

    public long get(String key, long def) {
        final String value = metadata.get(key);
        if (null == value) {
            return def;
        }
        return Long.parseLong(value);
    }

    public boolean get(String key, boolean def) {
        final String value = metadata.get(key);
        if (null == value) {
            return def;
        }
        return Boolean.parseBoolean(value);
    }

    public String get(String key, String def) {
        return metadata.getOrDefault(key, def);
    }

    public Peer peer() {
        return Peer.fromNum(peer);
    }
}
