package kadai.log
package json

import argonaut.{ EncodeJson, Json }
import scalaz.Show

import scala.reflect.ClassTag

/**
 * A Jsonable message. JsonLayout knows how to encode this payload.
 */
case class JsonMessage[A](value: A, json: List[Json.JsonAssoc]) extends LogMessage[A]

object JsonMessage extends LowPriorityImplicits {
  val Default = "msg" // preserves backwards-compatibility

  // need to fully qualify the JSON objects with a root qualifier due to limitations in ElasticSearch indexing
  trait Qualified[A] {
    def name: String
  }

  object Qualified {
    def apply[A: Qualified]: Qualified[A] = implicitly[Qualified[A]]

    def by[A](s: String): Qualified[A] =
      new Qualified[A] {
        def name = s
      }

    def field[A: EncodeJson: JsonMessage.Qualified](a: A): Json.JsonAssoc =
      JsonMessage.Qualified[A].name -> implicitly[EncodeJson[A]].apply(a)
  }

  def unqualified[A] =
    new Qualified[A] {
      def name = Default
    }

  implicit def StringQualifier =
    new Qualified[String] {
      def name = Default
    }
}

sealed trait LowPriorityImplicits {
  implicit def defaultQualified[A: ClassTag] =
    new JsonMessage.Qualified[A] {
      def name = implicitly[ClassTag[A]].runtimeClass.getSimpleName
    }
}