package kadai
package log
package json

import argonaut._
import Argonaut._
import scalaz.Show
import scalaz.syntax.id._

/**
 * Mixin if you want your class to be able to log structured objects in
 * JSON format. You will need an argonaut.EncodeJson instance in implicit
 * scope for the object you try and log.
 */
trait JsonLogging extends Logging

object JsonLogging extends JsonLoggingInstances

trait JsonLoggingInstances {
  implicit def QualifiedEncodeJsonLogWriter[A: EncodeJson: JsonMessage.Qualified]: LogWriter[A] =
    new LogWriter[A] {
      def apply(a: => A) = JsonMessage(a, List(JsonMessage.Qualified.field(a)))
    }

  implicit def TupleLogWriter[A: EncodeJson: JsonMessage.Qualified, B: EncodeJson: JsonMessage.Qualified]: LogWriter[(A, B)] =
    new LogWriter[(A, B)] {
      def apply(ab: => (A, B)) = ab |> {
        case (a, b) => JsonMessage(ab, List(JsonMessage.Qualified.field(a), JsonMessage.Qualified.field(b)))
      }
    }

  implicit val EncodeInvalid: EncodeJson[Invalid] =
    EncodeJson {
      case Invalid.Err(t) => ("error" := t.asJson) ->: jEmptyObject
      case Invalid.Message(m) => ("error" := m.asJson) ->: jEmptyObject
      case Invalid.Composite(i) => ("errors" := i.list.asJson) ->: jEmptyObject
      case Invalid.Zero => jEmptyObject
    }

  implicit val EncodeThrowable: EncodeJson[Throwable] =
    EncodeJson[Throwable] { t =>
      { "message" := Option(t.getMessage) } ->:
        { "class" := t.getClass.getName } ->:
        { "stacktrace" := t.getStackTrace.toVector } ->:
        { "cause" := Option(t.getCause).filter { _ != t } } ->:
        Json.jEmptyObject
    }

  implicit val EncodeStackTraceElement: EncodeJson[StackTraceElement] =
    implicitly[EncodeJson[String]].contramap { _.toString }
}
