package io.appmetrica.gradle.nologs

import com.android.build.gradle.LibraryExtension
import com.android.build.gradle.LibraryPlugin
import com.android.build.gradle.api.LibraryVariant
import org.gradle.api.Action
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.Task
import org.gradle.api.tasks.TaskProvider
import org.gradle.kotlin.dsl.create
import org.gradle.kotlin.dsl.getByType
import org.gradle.kotlin.dsl.hasPlugin
import org.gradle.kotlin.dsl.register

class NoLogsPlugin : Plugin<Project> {

    companion object {
        const val EXTENSION_NAME = "noLogs"
        private const val TASK_GROUP_NAME = "no logs"
        private const val NO_LOGS_TASK_NAME_PREFIX = "runNoLogsPlugin"
        private const val REVERSE_NO_LOGS_TASK_NAME_PREFIX = "runReverseNoLogsPlugin"
    }

    override fun apply(project: Project) {
        check(project.plugins.hasPlugin(LibraryPlugin::class)) { "Requires 'com.android.library' plugin" }

        val extension = project.extensions.create<NoLogsExtension>(EXTENSION_NAME)

        val noLogsTask = project.tasks.register(NO_LOGS_TASK_NAME_PREFIX)
        val reverseNoLogsTask = project.tasks.register(REVERSE_NO_LOGS_TASK_NAME_PREFIX)
        project.extensions.getByType<LibraryExtension>().libraryVariants.configureEach {
            if (extension.shouldRemoveLogs(this)) {
                createTask(project, this, extension).also {
                    noLogsTask.configure { dependsOn(it) }
                }
                val reverseTask = createReverseTask(project, this, extension)
                reverseNoLogsTask.configure { dependsOn(reverseTask) }
                configureVariantAssemble(project, this, extension, reverseTask)
            }
        }
    }

    private fun configureVariantAssemble(
        project: Project,
        variant: LibraryVariant,
        noLogsExtension: NoLogsExtension,
        reverseTask: TaskProvider<out Task>
    ) {
        val configuringAction: Action<Task> = Action {
            doFirst {
                LoggerCommenter.commentLoggers(
                    sources = variant.sourceSets,
                    loggerClasses = noLogsExtension.loggerClasses
                )
            }
            doLast {
                LoggerCommenter.uncommentLoggers(
                    sources = variant.sourceSets
                )
            }
            finalizedBy(reverseTask)
        }
        variant.javaCompileProvider.configure(configuringAction)
        project.plugins.withId("kotlin-android") {
            project.tasks.named("compile${variant.name.capitalize()}Kotlin", configuringAction)
        }
    }

    private fun createTask(project: Project, variant: LibraryVariant, noLogsExtension: NoLogsExtension) =
        project.tasks.register<RemoveLogsTask>("${NO_LOGS_TASK_NAME_PREFIX}For${variant.name.capitalize()}") {
            group = TASK_GROUP_NAME
            sources.set(variant.sourceSets)
            loggerClasses.set(noLogsExtension.loggerClasses)
            reverse.set(false)
        }

    private fun createReverseTask(project: Project, variant: LibraryVariant, noLogsExtension: NoLogsExtension) =
        project.tasks.register<RemoveLogsTask>("${REVERSE_NO_LOGS_TASK_NAME_PREFIX}For${variant.name.capitalize()}") {
            group = TASK_GROUP_NAME
            sources.set(variant.sourceSets)
            loggerClasses.set(noLogsExtension.loggerClasses)
            reverse.set(true)
        }
}
