package io.appmetrica.gradle.nologs

import com.android.builder.model.SourceProvider
import java.io.File

private const val COMMENT_START = "/*no-logs "
private const val COMMENT_END = " !no-logs*/"
private const val COMMENTED_PATTERN = "/\\*no-logs [\\s\\S]*? !no-logs\\*/"

object LoggerCommenter {

    fun commentLoggers(sources: List<SourceProvider>, loggerClasses: List<String>) {
        loggerClasses.forEach { logClass ->
            sources.forEachFile { file ->
                file.writeText(commentLogger(file.readText(), logClass, file.extension == "java"))
            }
        }
    }

    fun uncommentLoggers(sources: List<SourceProvider>) {
        sources.forEachFile { file ->
            file.writeText(uncommentLogger(file.readText()))
        }
    }

    private fun commentLogger(text: String, logClass: String, isJava: Boolean): String {
        val loggerCallRegexp = "$logClass\\s*\\.\\w+?\\s*\\(".toRegex(RegexOption.MULTILINE)
        var newText = text
        var startPosition = 0
        var matchResult = loggerCallRegexp.find(newText, startIndex = startPosition)
        while (matchResult != null) {
            var bracketCount = 1
            var position = matchResult.range.last + 1
            while (bracketCount != 0 && position != newText.length) {
                when (newText[position]) {
                    '(' -> bracketCount += 1
                    ')' -> bracketCount -= 1
                }
                position += 1
            }
            if (isJava) {
                while (position != newText.length && "${newText[position]}".isBlank()) {
                    position += 1
                }
                if (position != newText.length && newText[position] == ';') {
                    position += 1
                }
            }

            newText = newText.substring(0, matchResult.range.first) +
                COMMENT_START +
                newText.substring(matchResult.range.first, position) +
                COMMENT_END +
                newText.substring(position)
            startPosition = matchResult.range.last + COMMENT_START.length + COMMENT_END.length
            matchResult = if (startPosition < newText.length) {
                loggerCallRegexp.find(newText, startIndex = startPosition)
            } else {
                null
            }
        }
        return newText
    }

    private fun uncommentLogger(text: String) =
        COMMENTED_PATTERN.toRegex().replace(text) { uncomment(it.value) }

    private fun uncomment(text: String) =
        text.removeSurrounding(COMMENT_START, COMMENT_END)

    private fun List<SourceProvider>.forEachFile(action: (File) -> Unit) =
        forEach { source ->
            source.javaDirectories.forEach { dir ->
                dir.walkTopDown().filter(File::isFile).forEach(action)
            }
        }
}
