package io.appmetrica.gradle.common.plugins

import io.appmetrica.gradle.ktlint.KtLintPlugin
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.Task
import org.gradle.api.plugins.quality.Checkstyle
import org.gradle.api.plugins.quality.CheckstyleExtension
import org.gradle.api.plugins.quality.CheckstylePlugin
import org.gradle.api.plugins.quality.Pmd
import org.gradle.api.plugins.quality.PmdExtension
import org.gradle.api.plugins.quality.PmdPlugin
import org.gradle.api.tasks.TaskProvider
import org.gradle.kotlin.dsl.apply
import org.gradle.kotlin.dsl.configure
import org.gradle.kotlin.dsl.create
import org.gradle.kotlin.dsl.withType

class CodeQualityPlugin : Plugin<Project> {
    companion object {
        const val CODEQUALITY_TASK_GROUP = "verification"
        const val EXTENSION_NAME = "codequality"
        const val DEFAULT_CONFIG_DIR = "codequality"
        const val CHECKSTYLE_VERSION = "8.11" // https://github.com/checkstyle/checkstyle/releases
        const val PMD_VERSION = "5.5.4" // https://github.com/pmd/pmd/releases
    }

    override fun apply(target: Project): Unit = target.run {
        apply<CheckstylePlugin>()
        apply<PmdPlugin>()
        apply<KtLintPlugin>()

        val extension = extensions.create<CodeQualityExtension>(EXTENSION_NAME)
        extension.configDir.convention(rootProject.layout.projectDirectory.dir(DEFAULT_CONFIG_DIR))

        val checkstyleTask = configureCheckstyle(extension)
        val pmdTask = configurePmd(extension)
        val codequalityTask = tasks.register("codequality") {
            group = CODEQUALITY_TASK_GROUP
            description = "Run codestyle analysis for all classes"
            dependsOn(checkstyleTask, pmdTask, tasks.named("ktlintCheck"))
        }
        tasks.named("check") { dependsOn(codequalityTask) }
    }

    private fun Project.configureCheckstyle(extension: CodeQualityExtension): TaskProvider<Task> {
        configure<CheckstyleExtension> {
            toolVersion = CHECKSTYLE_VERSION
            configDirectory.set(extension.configDir)
            isShowViolations = true
            isIgnoreFailures = false
        }
        tasks.withType<Checkstyle> {
            group = CODEQUALITY_TASK_GROUP
        }
        return tasks.register("checkstyle") {
            group = CODEQUALITY_TASK_GROUP
            description = "Run Checkstyle analysis for all classes"
            dependsOn(tasks.withType<Checkstyle>())
        }
    }

    private fun Project.configurePmd(extension: CodeQualityExtension): TaskProvider<Task> {
        configure<PmdExtension> {
            toolVersion = PMD_VERSION
            incrementalAnalysis.set(false)
            isIgnoreFailures = false
        }
        tasks.withType<Pmd> {
            group = CODEQUALITY_TASK_GROUP
            ruleSetFiles = extension.configDir.files("pmd.xml")
            rulesMinimumPriority.set(2)
        }
        return tasks.register("pmd") {
            group = CODEQUALITY_TASK_GROUP
            description = "Run PMD analysis for all classes"
            dependsOn(tasks.withType<Pmd>())
        }
    }
}
