package io.appmetrica.gradle.android.codequality

import com.android.build.api.artifact.MultipleArtifact
import com.android.build.api.variant.ApplicationAndroidComponentsExtension
import com.android.build.api.variant.Component
import com.android.build.api.variant.LibraryAndroidComponentsExtension
import com.android.build.api.variant.Variant
import com.android.build.gradle.AppPlugin
import com.android.build.gradle.LibraryPlugin
import io.appmetrica.gradle.common.plugins.CodeQualityPlugin
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.plugins.quality.Checkstyle
import org.gradle.api.plugins.quality.Pmd
import org.gradle.api.tasks.TaskProvider
import org.gradle.kotlin.dsl.apply
import org.gradle.kotlin.dsl.configure
import org.gradle.kotlin.dsl.register
import org.gradle.kotlin.dsl.withType

@Suppress("UnstableApiUsage")
class AndroidCodeQualityPlugin : Plugin<Project> {

    companion object {
        private val DEFAULT_EXCLUDE_CLASSES = listOf("**/R.java", "**/BuildConfig.java")
    }

    override fun apply(project: Project) {
        project.apply<CodeQualityPlugin>() // id("appmetrica.codequality")

        // for id("com.android.library")
        project.plugins.withType<LibraryPlugin> {
            project.configure<LibraryAndroidComponentsExtension> {
                onVariants { project.configureForVariant(it) }
            }
        }
        // for id("com.android.application")
        project.plugins.withType<AppPlugin> {
            project.configure<ApplicationAndroidComponentsExtension> {
                onVariants { project.configureForVariant(it) }
            }
        }
    }

    private fun Project.configureForVariant(variant: Variant) {
        val checkstyleTask = configureCheckstyle(variant)
        val pmdTask = configurePmd(variant)
        val codequalityTask = project.tasks.register("codequality${variant.name.capitalize()}") {
            group = CodeQualityPlugin.CODEQUALITY_TASK_GROUP
            description = "Run codestyle analysis for ${variant.name} classes"
            dependsOn(checkstyleTask, pmdTask, tasks.named("lint${variant.name.capitalize()}"))
        }
        tasks.named("codequality") { dependsOn(codequalityTask) }
    }

    private fun Project.configureCheckstyle(variant: Variant): TaskProvider<Checkstyle> {
        val variants = getAllVariants(variant)
        return project.tasks.register<Checkstyle>("checkstyle${variant.name.capitalize()}") {
            description = "Run Checkstyle analysis for ${variant.name} classes"
            classpath = project.files(
                variants.map { it.artifacts.getAll(MultipleArtifact.ALL_CLASSES_DIRS) }
            )
            source(variants.map { it.sources.java.all })
            exclude(DEFAULT_EXCLUDE_CLASSES)
        }
    }

    private fun Project.configurePmd(variant: Variant): TaskProvider<Pmd> {
        val variants = getAllVariants(variant)
        return tasks.register<Pmd>("pmd${variant.name.capitalize()}") {
            description = "Run PMD analysis for ${variant.name} classes"
            source(variants.map { it.sources.java.all })
            include("**/*.java")
            exclude(DEFAULT_EXCLUDE_CLASSES)
        }
    }

    private fun getAllVariants(variant: Variant): List<Component> =
        listOfNotNull(variant, variant.unitTest)
}
