package io.appmetrica.gradle.aarcheck.tasks

import io.appmetrica.gradle.aarcheck.AarCheckException
import io.appmetrica.gradle.aarcheck.agp.AndroidLibraryVariant
import io.appmetrica.gradle.aarcheck.dependencies.DependenciesExtractor
import io.appmetrica.gradle.aarcheck.utils.DiffUtils
import org.gradle.api.DefaultTask
import org.gradle.api.file.RegularFileProperty
import org.gradle.api.provider.Property
import org.gradle.api.tasks.InputFile
import org.gradle.api.tasks.Internal
import org.gradle.api.tasks.TaskAction

abstract class DependenciesCheck : DefaultTask() {
    @get:Internal
    abstract val androidLibraryVariant: Property<AndroidLibraryVariant>

    @get:InputFile
    abstract val dependenciesDumpFile: RegularFileProperty

    @TaskAction
    fun check() {
        logger.info("Check dependencies for ${androidLibraryVariant.get().name}")
        val dumpFile = dependenciesDumpFile.get().asFile
        if (!dumpFile.exists()) {
            throw AarCheckException(
                "Dependencies check failed. Not found file ${dumpFile.canonicalPath}. " +
                    "You can run :dependenciesDump task to generate dependencies dump"
            )
        }

        val dump = dumpFile.readText()
        val expected = DependenciesExtractor.extract(androidLibraryVariant.get())
        val diff = DiffUtils.diff(dump, expected, "dump dependencies", "cur dependencies")
        if (diff != null) {
            logger.error("Diff:\n$diff")
            throw AarCheckException(
                "Dependencies check failed. See diff above. " +
                    "You can run :dependenciesDump task to overwrite dependencies dump"
            )
        }
    }
}
