package io.appmetrica.gradle.aarcheck.manifest

import io.appmetrica.gradle.aarcheck.AarCheckException
import io.appmetrica.gradle.aarcheck.AarCheckPlugin
import io.appmetrica.gradle.aarcheck.utils.DiffUtils
import org.slf4j.LoggerFactory
import java.io.File

class ManifestChecker(
    private val fullCheck: Boolean = false,
    private val versionCode: String? = null
) {
    private val logger = LoggerFactory.getLogger(AarCheckPlugin.PLUGIN_NAME)

    fun check(aarFile: File, manifestDumpFile: File) {
        if (!manifestDumpFile.exists()) {
            throw AarCheckException(
                "Manifest check failed. Not found file ${manifestDumpFile.canonicalPath}. " +
                    "You can run :manifestDump task to generate AndroidManifest dump"
            )
        }

        val manifestDump = if (fullCheck) {
            ManifestExtractor.fromTemplate(
                manifestDumpFile,
                listOfNotNull(
                    versionCode?.let { ManifestExtractor.VERSION_CODE_TOKEN to it }
                ).toMap()
            )
        } else {
            manifestDumpFile.readText()
        }
        val manifest = ManifestExtractor.extract(aarFile, asTemplate = !fullCheck)
        val diff = DiffUtils.diff(manifestDump, manifest, "dump manifest", "cur manifest")
        if (diff != null) {
            logger.error("Diff:\n$diff")
            throw AarCheckException(
                "Manifest check failed. See diff above. " +
                    "You can run :manifestDump task to overwrite AndroidManifest dump"
            )
        }
    }
}
