/*
 * Version for Android
 * © 2022
 * You may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * https://yandex.com/legal/appmetrica_sdk_agreement/
 */

package io.appmetrica.gradle.aarcheck.kotlinassertions

import io.appmetrica.gradle.aarcheck.AarCheckException
import io.appmetrica.gradle.aarcheck.utils.NamesUtils
import io.appmetrica.gradle.aarcheck.utils.extractEntry
import io.appmetrica.gradle.aarcheck.utils.isClass
import javassist.ClassPool
import javassist.bytecode.ConstPool
import java.io.File
import java.util.jar.JarEntry
import java.util.jar.JarFile
import java.util.zip.ZipFile

object ForbiddenMethodsChecker {

    fun check(aarFile: File, forbiddenMethods: Map<String, List<String>>) {
        if (forbiddenMethods.isEmpty()) return

        ZipFile(aarFile).use { zipFile ->
            JarFile(zipFile.extractEntry("classes.jar")).use { jarFile ->
                val classPool = ClassPool()
                classPool.appendClassPath(jarFile.name)

                for (jarEntry in jarFile.entries().toList().filter(JarEntry::isClass)) {
                    val ctClass = classPool.get(NamesUtils.getFullClassName(jarEntry.name))

                    val constPool = ctClass.classFile.constPool

                    for (index in 1 until constPool.size) {
                        if (constPool.getTag(index) != ConstPool.CONST_Methodref) continue
                        val className = constPool.getMethodrefClassName(index)
                        val methods = forbiddenMethods[className] ?: continue
                        if (constPool.getMethodrefName(index) in methods) {
                            throw AarCheckException(
                                "Found forbidden method - $className.${constPool.getMethodrefName(index)} " +
                                    "in class - ${ctClass.name}"
                            )
                        }
                    }
                }
            }
        }
    }
}
