package io.appmetrica.gradle.aarcheck.api.parser.javassist

import io.appmetrica.gradle.aarcheck.api.Class
import io.appmetrica.gradle.aarcheck.api.ClassType
import javassist.CtClass
import javassist.Modifier
import javassist.bytecode.AccessFlag
import javassist.bytecode.SignatureAttribute

object JavassistClass {
    fun from(ctClass: CtClass): Class {
        val signature = ctClass.genericSignature?.let(SignatureAttribute::toClassSignature)
        val superClass = signature?.superClass?.let(JavassistType::from)
            ?: ctClass.classFile.superclass.let(JavassistType::from)
        val clazz = Class(
            modifier = Utils.getAccessModifier(ctClass.modifiers),
            type = when {
                ctClass.isAnnotation -> ClassType.ANNOTATION
                ctClass.isInterface -> ClassType.INTERFACE
                ctClass.isEnum -> ClassType.ENUM
                else -> ClassType.CLASS
            },
            packageName = ctClass.packageName,
            name = ctClass.simpleName,
            genericParameters = signature?.parameters?.map(JavassistTypeParameter::from) ?: emptyList(),
            superClass = superClass.takeUnless { it.fullName == "java.lang.Object" },
            interfaces = signature?.interfaces?.map(JavassistType::from)
                ?: ctClass.classFile.interfaces.map(JavassistType::from),

            isSynthetic = (ctClass.modifiers and AccessFlag.SYNTHETIC) != 0,
            isAbstract = Modifier.isAbstract(ctClass.modifiers),
            isStatic = Modifier.isStatic(ctClass.modifiers),
            isFinal = Modifier.isFinal(ctClass.modifiers),

            fields = emptyList(),
            methods = emptyList()
        )
        return clazz.copy(
            fields = ctClass.declaredFields.map { JavassistField.from(clazz, it) },
            methods = ctClass.declaredBehaviors.map { JavassistMethod.from(clazz, it) }
        )
    }
}
