package io.appmetrica.gradle.aarcheck.api

import io.appmetrica.gradle.aarcheck.api.parser.javassist.JavassistParser
import io.appmetrica.gradle.aarcheck.utils.MappingFile
import io.appmetrica.gradle.aarcheck.utils.NamesUtils
import java.io.File

object ApiGenerator {
    fun generateTo(aarFile: File, apiFile: File, mappingFile: File? = null) {
        apiFile.parentFile.mkdirs()

        apiFile.bufferedWriter().use {
            generateTo(aarFile, it, mappingFile)
        }
    }

    fun generate(aarFile: File, mappingFile: File? = null): String {
        val api = StringBuilder()
        generateTo(aarFile, api, mappingFile)
        return api.toString()
    }

    fun generateTo(aarFile: File, appender: Appendable, mappingFile: File? = null) {
        val mapping = MappingFile(mappingFile)
        val classes = JavassistParser().parseAar(aarFile)
        val api = classes.getApiClasses(mapping).map { clazz ->
            clazz.copy(
                fields = clazz.fields.getApiFields(),
                methods = clazz.methods.getApiMethods()
            )
        }

        ApiSerializer(
            mappingFile = mapping
        ).writeTo(api, appender)
    }
}

private fun List<Class>.getApiClasses(mappingFile: MappingFile): List<Class> = this
    .filterNot { it.isSynthetic || it.isAnonim }
    .filterNot { it.isObfuscated(mappingFile) }

private fun Class.isObfuscated(mappingFile: MappingFile): Boolean {
    return when (mappingFile.hasSymbols) {
        true -> mappingFile.isObfuscatedClass(fullName)
        false -> NamesUtils.isObfuscatedClassName(name)
    }
}

private fun List<Field>.getApiFields(): List<Field> = this
    .filterNot { it.isSynthetic }
    .filter { it.modifier in listOf(AccessModifier.PUBLIC, AccessModifier.PROTECTED) }

private fun List<Method>.getApiMethods(): List<Method> = this
    .filterNot { it.isSynthetic }
    .filter { it.modifier in listOf(AccessModifier.PUBLIC, AccessModifier.PROTECTED) }
