package io.appmetrica.gradle.aarcheck.api

import io.appmetrica.gradle.aarcheck.AarCheckException
import io.appmetrica.gradle.aarcheck.AarCheckPlugin
import io.appmetrica.gradle.aarcheck.api.parser.javassist.JavassistParser
import io.appmetrica.gradle.aarcheck.utils.DiffUtils
import org.slf4j.LoggerFactory
import java.io.File

object ApiChecker {
    private val logger = LoggerFactory.getLogger(AarCheckPlugin.PLUGIN_NAME)

    fun check(aarFile: File, apiFile: File, mappingFile: File? = null) {
        if (!apiFile.exists()) {
            throw AarCheckException(
                "API check failed. Not found file ${apiFile.canonicalPath}. " +
                    "You can run :apiDump task to generate API declarations"
            )
        }

        val dumpApi = apiFile.readText()
        val api = ApiGenerator.generate(aarFile, mappingFile)
        val diff = DiffUtils.diff(dumpApi, api, "dump api", "cur api")
        if (diff != null) {
            logger.error("Diff:\n$diff")
            throw AarCheckException(
                "API check failed. See diff above. " +
                    "You can run :apiDump task to overwrite API declarations"
            )
        }
    }

    fun checkPackage(
        aarFile: File,
        packageName: String? = null
    ) {
        if (packageName == null) {
            throw AarCheckException(
                "Namespace is not set."
            )
        }
        val oddClasses = JavassistParser().parseAar(aarFile)
            .filterNot { it.packageName?.startsWith(packageName) ?: false }

        if (oddClasses.isNotEmpty()) {
            throw AarCheckException(
                "API check failed. Found classes not from package $packageName."
            )
        }
    }
}
