package io.appmetrica.gradle.aarcheck.agp

import com.android.build.api.artifact.SingleArtifact
import com.android.build.api.variant.LibraryVariant
import com.android.build.gradle.LibraryExtension
import io.appmetrica.gradle.aarcheck.AarCheckBuildTypeExtension
import io.appmetrica.gradle.aarcheck.AarCheckFlavorExtension
import org.gradle.api.Project
import org.gradle.api.artifacts.ResolvedDependency
import org.gradle.api.file.RegularFile
import org.gradle.api.provider.Provider
import org.gradle.kotlin.dsl.getByType

class AndroidLibraryVariant(private val project: Project, private val original: LibraryVariant) {
    val name: String
        get() = original.name

    val isAarCheckEnabled: Boolean = isEnabled()

    val versionCode: Provider<String> = project.provider {
        original.artifacts.getBuiltArtifactsLoader().load(project.files(aarFile))
            ?.elements?.single()?.versionCode?.toString()
    }

    val aarFile: Provider<RegularFile> = original.artifacts.get(SingleArtifact.AAR)

    val mappingFile: Provider<RegularFile> = original.artifacts.get(SingleArtifact.OBFUSCATION_MAPPING_FILE)

    fun getDependencies(): Set<String> {
        return project.configurations.findByName("${original.name}RuntimeClasspath")
            ?.resolvedConfiguration
            ?.firstLevelModuleDependencies
            ?.flatMap { getDependencies(it) }
            ?.toSortedSet() ?: emptySet()
    }

    private fun isEnabled(): Boolean {
        val androidLibraryExtension = project.extensions.getByType<LibraryExtension>()

        var enabled = true

        val buildTypeAarCheck = androidLibraryExtension.buildTypes.getByName(original.buildType!!)
            .extensions.getByType<AarCheckBuildTypeExtension>()
        enabled = enabled && buildTypeAarCheck.enabled

        original.productFlavors.forEach { (_, name) ->
            val flavorAarCheck = androidLibraryExtension.productFlavors.getByName(name)
                .extensions.getByType<AarCheckFlavorExtension>()
            enabled = enabled && flavorAarCheck.enabled
        }

        return enabled
    }

    private fun getDependencies(dependency: ResolvedDependency): List<String> =
        listOf(dependency.name) + dependency.children.flatMap { getDependencies(it) }
}
