package io.appmetrica.gradle.aarcheck

import io.appmetrica.gradle.aarcheck.agp.AndroidLibraryVariant
import io.appmetrica.gradle.aarcheck.tasks.ApiCheck
import io.appmetrica.gradle.aarcheck.tasks.ApiDump
import io.appmetrica.gradle.aarcheck.tasks.DependenciesCheck
import io.appmetrica.gradle.aarcheck.tasks.DependenciesDump
import io.appmetrica.gradle.aarcheck.tasks.ImportsCheck
import io.appmetrica.gradle.aarcheck.tasks.ManifestCheck
import io.appmetrica.gradle.aarcheck.tasks.ManifestDump
import io.appmetrica.gradle.aarcheck.tasks.MethodsCheck
import io.appmetrica.gradle.aarcheck.tasks.ModuleCheck
import io.appmetrica.gradle.aarcheck.tasks.ModuleDump
import io.appmetrica.gradle.aarcheck.tasks.PomCheck
import io.appmetrica.gradle.aarcheck.tasks.PomDump
import io.appmetrica.gradle.aarcheck.tasks.ProguardCheck
import io.appmetrica.gradle.aarcheck.tasks.ProguardDump
import io.appmetrica.gradle.aarcheck.tasks.TaskManager
import io.appmetrica.gradle.aarcheck.tasks.dependsOn
import org.gradle.api.Project
import org.gradle.api.Task
import org.gradle.api.file.Directory
import org.gradle.api.publish.maven.tasks.AbstractPublishToMaven
import org.gradle.api.publish.maven.tasks.GenerateMavenPom
import org.gradle.api.publish.tasks.GenerateModuleMetadata
import org.gradle.api.tasks.TaskProvider
import org.gradle.kotlin.dsl.withType

class TaskCreator(private val project: Project) {

    private val taskManager = TaskManager(project)
    private val dumpDir: Directory = project.layout.projectDirectory.dir("api")

    private fun aarDumpTask(): TaskProvider<Task> = taskManager.registerIfNotExists("aarDump")

    private fun aarDumpTaskForVariant(variantName: String): TaskProvider<Task> {
        return taskManager.registerIfNotExists("${variantName}AarDump").also {
            aarDumpTask().dependsOn(it)
        }
    }

    private fun aarCheckTask(): TaskProvider<Task> = taskManager.registerIfNotExists("aarCheck")

    private fun aarCheckTaskForVariant(variantName: String): TaskProvider<Task> {
        val publishTaskPrefix = "publish${variantName.capitalize()}PublicationTo"
        return taskManager.registerIfNotExists("${variantName}AarCheck").also {
            aarCheckTask().dependsOn(it)
            project.tasks.withType<AbstractPublishToMaven>().configureEach {
                if (name.startsWith(publishTaskPrefix)) {
                    dependsOn(it)
                }
            }
        }
    }

    fun createApiTasks(variant: AndroidLibraryVariant, namespace: String?) {
        val dumpFile = dumpDir.file("${variant.name}.api")

        val dumpTask = taskManager.registerIfNotExists("apiDump")
        val variantDumpTask = taskManager.register<ApiDump>("${variant.name}ApiDump") {
            aarFile.set(variant.aarFile)
            mappingFile.set(variant.mappingFile)
            apiDumpFile.set(dumpFile)
        }
        dumpTask.dependsOn(variantDumpTask)
        aarDumpTaskForVariant(variant.name).dependsOn(variantDumpTask)

        val checkTask = taskManager.registerIfNotExists("apiCheck")
        val variantCheckTask = taskManager.register<ApiCheck>("${variant.name}ApiCheck") {
            aarFile.set(variant.aarFile)
            mappingFile.set(variant.mappingFile)
            apiDumpFile.set(dumpFile)
            packageName.set(namespace)
        }
        checkTask.dependsOn(variantCheckTask)
        aarCheckTaskForVariant(variant.name).dependsOn(variantCheckTask)
    }

    fun createDependenciesTasks(variant: AndroidLibraryVariant) {
        val dumpFile = dumpDir.file("${variant.name}Dependencies.txt")

        val dumpTask = taskManager.registerIfNotExists("dependenciesDump")
        val variantDumpTask = taskManager.register<DependenciesDump>("${variant.name}DependenciesDump") {
            androidLibraryVariant.set(variant)
            dependenciesDumpFile.set(dumpFile)
            outputs.upToDateWhen { false }
        }
        dumpTask.dependsOn(variantDumpTask)
        aarDumpTaskForVariant(variant.name).dependsOn(variantDumpTask)

        val checkTask = taskManager.registerIfNotExists("dependenciesCheck")
        val variantCheckTask = taskManager.register<DependenciesCheck>("${variant.name}DependenciesCheck") {
            androidLibraryVariant.set(variant)
            dependenciesDumpFile.set(dumpFile)
            outputs.upToDateWhen { false }
        }
        checkTask.dependsOn(variantCheckTask)
        aarCheckTaskForVariant(variant.name).dependsOn(variantCheckTask)
    }

    fun createManifestTasks(variant: AndroidLibraryVariant) {
        val dumpFile = dumpDir.file("${variant.name}Manifest.xml")

        val dumpTask = taskManager.registerIfNotExists("manifestDump")
        val variantDumpTask = taskManager.register<ManifestDump>("${variant.name}ManifestDump") {
            aarFile.set(variant.aarFile)
            manifestDumpFile.set(dumpFile)
        }
        dumpTask.dependsOn(variantDumpTask)
        aarDumpTaskForVariant(variant.name).dependsOn(variantDumpTask)

        val checkTask = taskManager.registerIfNotExists("manifestCheck")
        val variantCheckTask = taskManager.register<ManifestCheck>("${variant.name}ManifestCheck") {
            aarFile.set(variant.aarFile)
            versionCode.set(variant.versionCode)
            manifestDumpFile.set(dumpFile)
        }
        checkTask.dependsOn(variantCheckTask)
        aarCheckTaskForVariant(variant.name).dependsOn(variantCheckTask)
    }

    fun createModuleTasks(variant: AndroidLibraryVariant) {
        val lazyGenerateTask = project.tasks.withType<GenerateModuleMetadata>()
            .matching { it.name == "generateMetadataFileFor${variant.name.capitalize()}Publication" }
        val dumpFile = dumpDir.file("${variant.name}Module.json")

        val dumpTask = taskManager.registerIfNotExists("moduleDump")
        val variantDumpTask = taskManager.register<ModuleDump>("${variant.name}ModuleDump") {
            moduleFile.set(lazyGenerateTask.single().outputFile)
            moduleDumpFile.set(dumpFile)
        }
        dumpTask.dependsOn(variantDumpTask)
        aarDumpTaskForVariant(variant.name).dependsOn(variantDumpTask)

        val checkTask = taskManager.registerIfNotExists("moduleCheck")
        val variantCheckTask = taskManager.register<ModuleCheck>("${variant.name}ModuleCheck") {
            moduleFile.set(lazyGenerateTask.single().outputFile)
            moduleDumpFile.set(dumpFile)
        }
        checkTask.dependsOn(variantCheckTask)
        aarCheckTaskForVariant(variant.name).dependsOn(variantCheckTask)
    }

    fun createPomTasks(variant: AndroidLibraryVariant) {
        val lazyGenerateTask = project.tasks.withType<GenerateMavenPom>()
            .matching { it.name == "generatePomFileFor${variant.name.capitalize()}Publication" }
        val dumpFile = dumpDir.file("${variant.name}Pom.xml")

        val dumpTask = taskManager.registerIfNotExists("pomDump")
        val variantDumpTask = taskManager.register<PomDump>("${variant.name}PomDump") {
            pomFile.set(lazyGenerateTask.single().destination)
            pomDumpFile.set(dumpFile)
            dependsOn(lazyGenerateTask)
        }
        dumpTask.dependsOn(variantDumpTask)
        aarDumpTaskForVariant(variant.name).dependsOn(variantDumpTask)

        val checkTask = taskManager.registerIfNotExists("pomCheck")
        val variantCheckTask = taskManager.register<PomCheck>("${variant.name}PomCheck") {
            pomFile.set(lazyGenerateTask.single().destination)
            pomDumpFile.set(dumpFile)
            dependsOn(lazyGenerateTask)
        }
        checkTask.dependsOn(variantCheckTask)
        aarCheckTaskForVariant(variant.name).dependsOn(variantCheckTask)
    }

    fun createProguardTasks(variant: AndroidLibraryVariant) {
        val dumpFile = dumpDir.file("${variant.name}Proguard.txt")

        val dumpTask = taskManager.registerIfNotExists("proguardDump")
        val variantDumpTask = taskManager.register<ProguardDump>("${variant.name}ProguardDump") {
            aarFile.set(variant.aarFile)
            proguardDumpFile.set(dumpFile)
        }
        dumpTask.dependsOn(variantDumpTask)
        aarDumpTaskForVariant(variant.name).dependsOn(variantDumpTask)

        val checkTask = taskManager.registerIfNotExists("proguardCheck")
        val variantCheckTask = taskManager.register<ProguardCheck>("${variant.name}ProguardCheck") {
            aarFile.set(variant.aarFile)
            proguardDumpFile.set(dumpFile)
        }
        checkTask.dependsOn(variantCheckTask)
        aarCheckTaskForVariant(variant.name).dependsOn(variantCheckTask)
    }

    fun createImportTasks(variant: AndroidLibraryVariant, forbiddenImports: List<String>) {
        val checkTask = taskManager.registerIfNotExists("importCheck")
        val variantCheckTask = taskManager.register<ImportsCheck>("${variant.name}ImportCheck") {
            aarFile.set(variant.aarFile)
            this.forbiddenImports.set(forbiddenImports)
        }
        checkTask.dependsOn(variantCheckTask)
        aarCheckTaskForVariant(variant.name).dependsOn(variantCheckTask)
    }

    fun createForbiddenMethodsTasks(variant: AndroidLibraryVariant, forbiddenMethods: Map<String, List<String>>) {
        val checkTask = taskManager.registerIfNotExists("forbiddenMethodsCheck")
        val variantCheckTask = taskManager.register<MethodsCheck>("${variant.name}ForbiddenMethodsCheck") {
            aarFile.set(variant.aarFile)
            this.forbiddenMethods.set(forbiddenMethods)
        }
        checkTask.dependsOn(variantCheckTask)
        aarCheckTaskForVariant(variant.name).dependsOn(variantCheckTask)
    }
}
