package io.appmetrica.gradle.aarcheck

import io.appmetrica.gradle.aarcheck.agp.AndroidLibraryVariant
import io.appmetrica.gradle.aarcheck.agp.AndroidPluginHelper
import org.gradle.api.Plugin
import org.gradle.api.Project

class AarCheckPlugin : Plugin<Project> {
    companion object {
        const val PLUGIN_NAME = "aar-check"
        const val EXTENSION_NAME = "aarCheck"
    }

    private lateinit var taskCreator: TaskCreator
    private lateinit var androidPluginHelper: AndroidPluginHelper

    override fun apply(project: Project) {
        taskCreator = TaskCreator(project)
        androidPluginHelper = AndroidPluginHelper(project)
        androidPluginHelper.requirePlugin()
        androidPluginHelper.addExtensionsToDsl()

        val extension = project.extensions.create(EXTENSION_NAME, AarCheckExtension::class.java)

        androidPluginHelper.configureEachVariant { variant ->
            if (variant.isAarCheckEnabled) {
                createTasksForVariant(extension, variant)
            }
        }
    }

    private fun createTasksForVariant(extension: AarCheckExtension, variant: AndroidLibraryVariant) {
        taskCreator.createApiTasks(variant, androidPluginHelper.getNamespace())
        if (extension.checkDependencies) {
            taskCreator.createDependenciesTasks(variant)
        }
        if (extension.checkManifest) {
            taskCreator.createManifestTasks(variant)
        }
        if (extension.checkModule) {
            taskCreator.createModuleTasks(variant)
        }
        if (extension.checkPom) {
            taskCreator.createPomTasks(variant)
        }
        if (extension.checkProguard) {
            taskCreator.createProguardTasks(variant)
        }
        if (extension.forbiddenImports.isNotEmpty()) {
            taskCreator.createImportTasks(variant, extension.forbiddenImports)
        }
        if (extension.forbiddenMethods.isNotEmpty()) {
            taskCreator.createForbiddenMethodsTasks(variant, extension.forbiddenMethods)
        }
    }
}
