/*
 * Decompiled with CFR 0.152.
 */
package io.airlift.security.pem;

import com.google.common.collect.MoreCollectors;
import com.google.common.io.Files;
import io.airlift.security.pem.PemReader;
import io.airlift.security.pem.PemWriter;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.Key;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Optional;
import java.util.regex.Matcher;
import javax.naming.InvalidNameException;
import javax.naming.ldap.LdapName;
import org.testng.Assert;
import org.testng.annotations.Test;

public class TestPemReader {
    private static final String CA_NAME = "OU=RootCA,O=Airlift,L=Palo Alto,ST=CA,C=US";
    private static final String CLIENT_NAME = "CN=Test User,OU=Server,O=Airlift,L=Palo Alto,ST=CA,C=US";
    private static final Optional<String> NO_PASSWORD = Optional.empty();
    private static final Optional<String> KEY_PASSWORD = Optional.of("airlift");

    @Test
    public void testLoadKeyStore() throws Exception {
        TestPemReader.testLoadKeyStore("rsa.client.crt", "rsa.client.pkcs8.key", NO_PASSWORD, CLIENT_NAME);
        TestPemReader.testLoadKeyStore("ec.client.crt", "ec.client.pkcs8.key", NO_PASSWORD, CLIENT_NAME);
        TestPemReader.testLoadKeyStore("dsa.client.crt", "dsa.client.pkcs8.key", NO_PASSWORD, CLIENT_NAME);
        TestPemReader.testLoadKeyStore("rsa.client.crt", "rsa.client.pkcs8.key.encrypted", KEY_PASSWORD, CLIENT_NAME);
        TestPemReader.testLoadKeyStore("ec.client.crt", "ec.client.pkcs8.key.encrypted", KEY_PASSWORD, CLIENT_NAME);
        TestPemReader.testLoadKeyStore("dsa.client.crt", "dsa.client.pkcs8.key.encrypted", KEY_PASSWORD, CLIENT_NAME);
        TestPemReader.testLoadKeyStore("rsa.client.pkcs8.pem.encrypted", "rsa.client.pkcs8.pem.encrypted", KEY_PASSWORD, CLIENT_NAME);
        TestPemReader.testLoadKeyStore("dsa.client.pkcs8.pem.encrypted", "dsa.client.pkcs8.pem.encrypted", KEY_PASSWORD, CLIENT_NAME);
        TestPemReader.testLoadKeyStore("ec.client.pkcs8.pem.encrypted", "ec.client.pkcs8.pem.encrypted", KEY_PASSWORD, CLIENT_NAME);
        TestPemReader.testLoadKeyStore("rsa.client.crt", "rsa.client.pkcs1.key", NO_PASSWORD, CLIENT_NAME);
        TestPemReader.testLoadKeyStore("ec.client.crt", "ec.client.pkcs1.key", NO_PASSWORD, CLIENT_NAME);
        TestPemReader.testLoadKeyStore("dsa.client.crt", "dsa.client.pkcs1.key", NO_PASSWORD, CLIENT_NAME);
        TestPemReader.testLoadKeyStore("rsa.client.pkcs8.pem.encrypted", "rsa.client.pkcs1.pem", NO_PASSWORD, CLIENT_NAME);
        TestPemReader.testLoadKeyStore("dsa.client.pkcs8.pem.encrypted", "dsa.client.pkcs1.pem", NO_PASSWORD, CLIENT_NAME);
        TestPemReader.testLoadKeyStore("ec.client.pkcs8.pem.encrypted", "ec.client.pkcs1.pem", NO_PASSWORD, CLIENT_NAME);
    }

    private static void testLoadKeyStore(String certFile, String keyFile, Optional<String> keyPassword, String expectedName) throws Exception {
        KeyStore keyStore = PemReader.loadKeyStore((File)TestPemReader.getResourceFile(certFile), (File)TestPemReader.getResourceFile(keyFile), keyPassword);
        TestPemReader.assertCertificateChain(keyStore, expectedName);
        Assert.assertNotNull((Object)keyStore.getCertificate("key"));
        Key key = keyStore.getKey("key", new char[0]);
        Assert.assertNotNull((Object)key);
        Assert.assertTrue((boolean)(key instanceof PrivateKey));
        PrivateKey privateKey = (PrivateKey)key;
        String encodedPrivateKey = PemWriter.writePrivateKey((PrivateKey)privateKey);
        Assert.assertEquals((Object)key, (Object)PemReader.loadPrivateKey((String)encodedPrivateKey, Optional.empty()));
    }

    @Test
    public void testLoadTrustStore() throws Exception {
        TestPemReader.assertCertificateChain(PemReader.loadTrustStore((File)TestPemReader.getResourceFile("rsa.ca.crt")), CA_NAME);
        TestPemReader.assertCertificateChain(PemReader.loadTrustStore((File)TestPemReader.getResourceFile("ec.ca.crt")), CA_NAME);
        TestPemReader.assertCertificateChain(PemReader.loadTrustStore((File)TestPemReader.getResourceFile("dsa.ca.crt")), CA_NAME);
    }

    @Test
    public void testLoadPublicKey() throws Exception {
        TestPemReader.testLoadPublicKey("rsa.client.crt", "rsa.client.pkcs8.pub");
        TestPemReader.testLoadPublicKey("rsa.client.crt", "rsa.client.pkcs1.pub");
        TestPemReader.testLoadPublicKey("ec.client.crt", "ec.client.pkcs8.pub");
        TestPemReader.testLoadPublicKey("dsa.client.crt", "dsa.client.pkcs8.pub");
    }

    private static void testLoadPublicKey(String certFile, String keyFile) throws Exception {
        File file = TestPemReader.getResourceFile(keyFile);
        Assert.assertTrue((boolean)PemReader.isPem((File)file));
        PublicKey publicKey = PemReader.loadPublicKey((File)file);
        Assert.assertNotNull((Object)publicKey);
        X509Certificate certificate = (X509Certificate)PemReader.readCertificateChain((File)TestPemReader.getResourceFile(certFile)).stream().collect(MoreCollectors.onlyElement());
        Assert.assertEquals((Object)publicKey, (Object)certificate.getPublicKey());
        String encodedPrivateKey = PemWriter.writePublicKey((PublicKey)publicKey);
        Assert.assertEquals((Object)publicKey, (Object)PemReader.loadPublicKey((String)encodedPrivateKey));
    }

    @Test
    public void testRsaPublicKeyPkcs1ToPkcs8() throws Exception {
        byte[] pkcs8 = TestPemReader.loadPublicKeyData("rsa.client.pkcs8.pub");
        byte[] pkcs1 = TestPemReader.loadPublicKeyData("rsa.client.pkcs1.pub");
        Assert.assertEquals((Object)PemReader.rsaPublicKeyPkcs1ToPkcs8((byte[])pkcs1), (Object)pkcs8);
    }

    @Test
    public void testRsaPkcs1ToPkcs8() throws Exception {
        byte[] pkcs8 = TestPemReader.loadPrivateKeyData("rsa.client.pkcs8.key");
        byte[] pkcs1 = TestPemReader.loadPrivateKeyData("rsa.client.pkcs1.key");
        Assert.assertEquals((Object)PemReader.rsaPkcs1ToPkcs8((byte[])pkcs1), (Object)pkcs8);
    }

    @Test
    public void testDsaPkcs1ToPkcs8() throws Exception {
        byte[] pkcs8 = TestPemReader.loadPrivateKeyData("dsa.client.pkcs8.key");
        byte[] pkcs1 = TestPemReader.loadPrivateKeyData("dsa.client.pkcs1.key");
        Assert.assertEquals((Object)PemReader.dsaPkcs1ToPkcs8((byte[])pkcs1), (Object)pkcs8);
    }

    @Test
    public void testEcPkcs1ToPkcs8() throws Exception {
        byte[] pkcs8 = TestPemReader.loadPrivateKeyData("ec.client.pkcs8.key");
        byte[] pkcs1 = TestPemReader.loadPrivateKeyData("ec.client.pkcs1.key");
        Assert.assertEquals((Object)PemReader.ecPkcs1ToPkcs8((byte[])pkcs1), (Object)pkcs8);
    }

    private static void assertCertificateChain(KeyStore keyStore, String expectedName) throws Exception {
        ArrayList<String> aliases = Collections.list(keyStore.aliases());
        Assert.assertEquals((int)aliases.size(), (int)1);
        Certificate certificate = keyStore.getCertificate(aliases.get(0));
        Assert.assertNotNull((Object)certificate);
        Assert.assertTrue((boolean)(certificate instanceof X509Certificate));
        X509Certificate x509Certificate = (X509Certificate)certificate;
        TestPemReader.assertX509Certificate(x509Certificate, expectedName);
        X509Certificate certificateCopy = (X509Certificate)PemReader.readCertificateChain((String)PemWriter.writeCertificate((X509Certificate)x509Certificate)).stream().collect(MoreCollectors.onlyElement());
        TestPemReader.assertX509Certificate(certificateCopy, expectedName);
    }

    private static void assertX509Certificate(X509Certificate x509Certificate, String expectedName) throws InvalidNameException {
        LdapName ldapName = new LdapName(x509Certificate.getSubjectX500Principal().getName());
        Assert.assertEquals((String)ldapName.toString(), (String)expectedName);
    }

    private static byte[] loadPrivateKeyData(String keyFile) throws IOException, KeyStoreException {
        File file = TestPemReader.getResourceFile(keyFile);
        Assert.assertTrue((boolean)PemReader.isPem((File)file));
        String privateKey = Files.asCharSource((File)file, (Charset)StandardCharsets.US_ASCII).read();
        Matcher matcher = PemReader.PRIVATE_KEY_PATTERN.matcher(privateKey);
        if (!matcher.find()) {
            throw new KeyStoreException("did not find a private key");
        }
        byte[] data = PemReader.base64Decode((String)matcher.group(2));
        return data;
    }

    private static byte[] loadPublicKeyData(String keyFile) throws IOException, KeyStoreException {
        File file = TestPemReader.getResourceFile(keyFile);
        Assert.assertTrue((boolean)PemReader.isPem((File)file));
        String privateKey = Files.asCharSource((File)file, (Charset)StandardCharsets.US_ASCII).read();
        Matcher matcher = PemReader.PUBLIC_KEY_PATTERN.matcher(privateKey);
        if (!matcher.find()) {
            throw new KeyStoreException("did not find a private key");
        }
        byte[] data = PemReader.base64Decode((String)matcher.group(2));
        return data;
    }

    private static File getResourceFile(String name) {
        URL resource = TestPemReader.class.getClassLoader().getResource(name);
        if (resource == null) {
            throw new IllegalArgumentException("Resource not found " + name);
        }
        return new File(resource.getFile());
    }
}

