/*
 * Decompiled with CFR 0.152.
 */
package io.airlift.log;

import com.google.common.collect.ImmutableSet;
import com.google.common.io.ByteStreams;
import com.google.common.io.MoreFiles;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import io.airlift.log.Format;
import io.airlift.log.LegacyRollingFileHandler;
import io.airlift.log.LogFileName;
import io.airlift.log.LogHistoryManager;
import io.airlift.units.DataSize;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Handler;
import java.util.logging.LogRecord;
import java.util.zip.GZIPOutputStream;
import javax.annotation.concurrent.GuardedBy;

final class RollingFileHandler
extends Handler {
    private static final int MAX_OPEN_NEW_LOG_ATTEMPTS = 100;
    private static final int MAX_BATCH_COUNT = 1024;
    private static final int MAX_BATCH_BYTES = Math.toIntExact(new DataSize(1.0, DataSize.Unit.MEGABYTE).toBytes());
    private static final String TEMP_PREFIX = ".tmp.";
    private static final String DELETED_PREFIX = ".deleted.";
    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("-yyyyMMdd.HHmmss");
    private static final byte[] POISON_MESSAGE = new byte[0];
    private final Path symlink;
    private final long maxFileSize;
    private final CompressionType compressionType;
    @GuardedBy(value="this")
    private Path currentOutputFile;
    @GuardedBy(value="this")
    private LogFileName currentOutputFileName;
    @GuardedBy(value="this")
    private OutputStream currentOutputStream;
    @GuardedBy(value="this")
    private long currentFileSize;
    private final LogHistoryManager historyManager;
    private final BlockingQueue<byte[]> queue = new ArrayBlockingQueue<byte[]>(1024);
    private final AtomicBoolean closed = new AtomicBoolean();
    private final Thread thread;
    private final ExecutorService compressionExecutor;

    public static RollingFileHandler createRollingFileHandler(String filename, DataSize maxFileSize, DataSize maxTotalSize, CompressionType compressionType, Format format) {
        RollingFileHandler handler = new RollingFileHandler(filename, maxFileSize, maxTotalSize, compressionType, format);
        handler.start();
        return handler;
    }

    private RollingFileHandler(String filename, DataSize maxFileSize, DataSize maxTotalSize, CompressionType compressionType, Format format) {
        Objects.requireNonNull(filename, "filename is null");
        Objects.requireNonNull(maxFileSize, "maxFileSize is null");
        Objects.requireNonNull(maxTotalSize, "maxTotalSize is null");
        Objects.requireNonNull(compressionType, "compressionType is null");
        this.maxFileSize = maxFileSize.toBytes();
        this.compressionType = compressionType;
        this.symlink = Paths.get(filename, new String[0]);
        this.thread = new Thread(this::logging);
        this.thread.setName("log-writer");
        this.thread.setDaemon(true);
        this.setFormatter(format.getFormatter());
        try {
            MoreFiles.createParentDirectories((Path)this.symlink, (FileAttribute[])new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        try {
            LegacyRollingFileHandler.recoverTempFiles(filename);
        }
        catch (IOException e) {
            this.reportError(null, e, 4);
        }
        if (Files.exists(this.symlink, new LinkOption[0])) {
            try {
                BasicFileAttributes attributes = Files.readAttributes(this.symlink, BasicFileAttributes.class, new LinkOption[0]);
                if (attributes.isDirectory()) {
                    throw new IllegalArgumentException("Log file is an existing directory: " + filename);
                }
                if (attributes.isRegularFile()) {
                    LocalDateTime createTime = LocalDateTime.ofInstant(attributes.creationTime().toInstant(), ZoneId.systemDefault()).withNano(0);
                    Path logFile = this.symlink.resolveSibling(this.symlink.getFileName() + DATE_TIME_FORMATTER.format(createTime) + "--" + UUID.randomUUID());
                    Files.move(this.symlink, logFile, StandardCopyOption.ATOMIC_MOVE);
                }
            }
            catch (IOException e) {
                throw new UncheckedIOException("Unable to update move legacy log file to a new file", e);
            }
        }
        RollingFileHandler.tryCleanupTempFiles(this.symlink);
        this.historyManager = new LogHistoryManager(this.symlink, maxTotalSize);
        try {
            this.rollFile();
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        this.compressionExecutor = compressionType != CompressionType.NONE ? Executors.newSingleThreadExecutor(new ThreadFactoryBuilder().setDaemon(true).setNameFormat("log-compression-%d").build()) : null;
    }

    private void start() {
        this.thread.start();
    }

    public synchronized Set<LogFileName> getFiles() {
        ImmutableSet.Builder files = ImmutableSet.builder().addAll(this.historyManager.getFiles());
        if (this.currentOutputFileName != null) {
            files.add((Object)this.currentOutputFileName);
        }
        return files.build();
    }

    @Override
    public void publish(LogRecord record) {
        byte[] message;
        if (this.closed.get()) {
            return;
        }
        if (!this.isLoggable(record)) {
            return;
        }
        try {
            message = this.getFormatter().format(record).getBytes(StandardCharsets.UTF_8);
        }
        catch (Exception e) {
            this.reportError(null, e, 5);
            return;
        }
        try {
            RollingFileHandler.putUninterruptibly(this.queue, message);
        }
        catch (Exception e) {
            this.reportError(null, e, 1);
        }
        if (this.closed.get()) {
            this.queue.remove(message);
        }
    }

    @Override
    public synchronized void flush() {
        if (this.currentOutputStream != null) {
            try {
                this.currentOutputStream.flush();
            }
            catch (Exception e) {
                this.reportError(null, e, 2);
            }
        }
    }

    @Override
    public void close() {
        this.closed.set(true);
        RollingFileHandler.putUninterruptibly(this.queue, POISON_MESSAGE);
        try {
            this.thread.join();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        this.queue.clear();
        if (this.compressionExecutor != null) {
            this.compressionExecutor.shutdown();
            try {
                this.compressionExecutor.awaitTermination(1L, TimeUnit.MINUTES);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void logging() {
        while (!this.closed.get() || !this.queue.isEmpty()) {
            this.processQueue();
        }
        RollingFileHandler rollingFileHandler = this;
        synchronized (rollingFileHandler) {
            try {
                this.currentOutputStream.flush();
            }
            catch (IOException e) {
                this.reportError("Could not flush output stream", e, 3);
            }
            try {
                this.currentOutputStream.close();
            }
            catch (IOException e) {
                this.reportError("Could not close output stream", e, 3);
            }
            this.currentOutputFile = null;
            this.currentOutputFileName = null;
            this.currentOutputStream = null;
            this.currentFileSize = 0L;
        }
        this.queue.clear();
    }

    private void processQueue() {
        List<Object> batch = new ArrayList<byte[]>(1024);
        boolean poisonMessageSeen = false;
        while (!this.closed.get() || !poisonMessageSeen) {
            if (this.queue.isEmpty()) {
                try {
                    batch.add(this.queue.take());
                }
                catch (InterruptedException interruptedException) {}
            } else {
                this.queue.drainTo(batch, 1024);
            }
            int poisonMessageIndex = RollingFileHandler.getPoisonMessageIndex(batch);
            if (poisonMessageIndex >= 0) {
                poisonMessageSeen = true;
                batch = batch.subList(0, poisonMessageIndex);
            }
            this.logMessageBatch(batch);
            batch.clear();
        }
    }

    private static int getPoisonMessageIndex(List<byte[]> messages) {
        for (int i = 0; i < messages.size(); ++i) {
            if (messages.get(i) != POISON_MESSAGE) continue;
            return i;
        }
        return -1;
    }

    private synchronized void logMessageBatch(List<byte[]> batch) {
        for (byte[] message : batch) {
            if (this.currentFileSize + (long)message.length > this.maxFileSize) {
                try {
                    this.rollFile();
                }
                catch (IOException e) {
                    this.currentFileSize = 0L;
                    this.reportError("Error rolling log file", e, 0);
                }
            }
            this.historyManager.pruneLogFilesIfNecessary(this.currentFileSize + (long)message.length);
            this.currentFileSize += (long)message.length;
            try {
                this.currentOutputStream.write(message);
            }
            catch (Exception e) {
                this.reportError(null, e, 1);
            }
        }
        this.flush();
    }

    private synchronized void rollFile() throws IOException {
        LogFileName newFileName = null;
        Path newFile = null;
        BufferedOutputStream newOutputStream = null;
        for (int i = 0; i < 100; ++i) {
            try {
                newFileName = LogFileName.generateNextLogFileName(this.symlink, this.compressionType.getExtension());
                newFile = this.symlink.resolveSibling(newFileName.getFileName());
                newOutputStream = new BufferedOutputStream(Files.newOutputStream(newFile, StandardOpenOption.CREATE_NEW), MAX_BATCH_BYTES);
                break;
            }
            catch (FileAlreadyExistsException fileAlreadyExistsException) {
                continue;
            }
        }
        if (newOutputStream == null) {
            throw new IOException("Could not create new a unique log file: " + newFile);
        }
        IOException exception = new IOException(String.format("Unable to %s log file", this.currentOutputStream == null ? "setup initial" : "roll"));
        if (this.currentOutputStream != null) {
            try {
                this.currentOutputStream.close();
            }
            catch (IOException e) {
                exception.addSuppressed(new IOException("Unable to close old output stream: " + this.currentOutputFile, e));
            }
            this.historyManager.addFile(this.currentOutputFile, this.currentOutputFileName, this.currentFileSize);
            if (this.compressionExecutor != null) {
                Path originalFile = this.currentOutputFile;
                LogFileName originalLogFileName = this.currentOutputFileName;
                long originalFileSize = this.currentFileSize;
                this.compressionExecutor.submit(() -> this.compressInternal(originalFile, originalLogFileName, originalFileSize));
            }
        }
        this.currentOutputFile = newFile;
        this.currentOutputFileName = newFileName;
        this.currentOutputStream = newOutputStream;
        this.currentFileSize = 0L;
        try {
            if (Files.exists(this.symlink, new LinkOption[0])) {
                Files.delete(this.symlink);
            }
            Files.createSymbolicLink(this.symlink, newFile, new FileAttribute[0]);
        }
        catch (IOException e) {
            exception.addSuppressed(new IOException("Unable to update symlink", e));
        }
        if (exception.getSuppressed().length > 0) {
            throw exception;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void compressInternal(Path originalFile, LogFileName originalLogFileName, long originalFileSize) {
        long compressedSize;
        RollingFileHandler.tryCleanupTempFiles(this.symlink);
        String compressionExtension = this.compressionType.getExtension().orElseThrow(IllegalStateException::new);
        Path tempFile = originalFile.resolveSibling(TEMP_PREFIX + originalFile.getFileName() + compressionExtension);
        try (InputStream input = Files.newInputStream(originalFile, new OpenOption[0]);
             GZIPOutputStream gzipOutputStream = new GZIPOutputStream(Files.newOutputStream(tempFile, new OpenOption[0]));){
            ByteStreams.copy((InputStream)input, (OutputStream)gzipOutputStream);
        }
        catch (IOException e) {
            this.reportError("Unable to compress log file", e, 0);
            return;
        }
        try {
            compressedSize = Files.size(tempFile);
        }
        catch (IOException e) {
            this.reportError("Unable to get size of compress log file", e, 0);
            return;
        }
        RollingFileHandler rollingFileHandler = this;
        synchronized (rollingFileHandler) {
            if (!this.historyManager.removeFile(originalFile)) {
                try {
                    Files.deleteIfExists(tempFile);
                }
                catch (IOException e) {
                    this.reportError("Unable to delete compress log file", e, 0);
                }
                return;
            }
            Path compressedFile = originalFile.resolveSibling(originalFile.getFileName() + compressionExtension);
            LogFileName compressedFileName = originalLogFileName.withCompression(compressedFile);
            try {
                Files.move(tempFile, compressedFile, StandardCopyOption.ATOMIC_MOVE);
            }
            catch (IOException e) {
                this.historyManager.addFile(originalFile, originalLogFileName, originalFileSize);
                try {
                    Files.deleteIfExists(tempFile);
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }
            this.historyManager.addFile(compressedFile, compressedFileName, compressedSize);
            try {
                Files.deleteIfExists(originalFile);
            }
            catch (IOException deleteException) {
                try {
                    Files.move(originalFile, originalFile.resolveSibling(DELETED_PREFIX + originalFile.getFileName()), StandardCopyOption.ATOMIC_MOVE);
                }
                catch (IOException ignored) {
                    this.reportError("Unable to delete original file after compression", deleteException, 0);
                }
            }
        }
    }

    private static void tryCleanupTempFiles(Path masterLogFile) {
        try {
            for (Path file : MoreFiles.listFiles((Path)masterLogFile.getParent())) {
                String fileNameWithoutPrefix;
                String fileName = file.getFileName().toString();
                if (fileName.startsWith(TEMP_PREFIX)) {
                    fileNameWithoutPrefix = fileName.substring(TEMP_PREFIX.length());
                } else {
                    if (!fileName.startsWith(DELETED_PREFIX)) continue;
                    fileNameWithoutPrefix = fileName.substring(DELETED_PREFIX.length());
                }
                if (!LogFileName.parseHistoryLogFileName(masterLogFile.getFileName().toString(), fileNameWithoutPrefix).isPresent()) continue;
                try {
                    Files.deleteIfExists(file);
                }
                catch (IOException iOException) {}
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private static <T> void putUninterruptibly(BlockingQueue<T> queue, T element) {
        boolean interrupted = false;
        while (true) {
            try {
                queue.put(element);
                return;
            }
            catch (InterruptedException e) {
                interrupted = true;
                continue;
            }
            break;
        }
        finally {
            if (interrupted) {
                Thread.currentThread().interrupt();
            }
        }
    }

    public static enum CompressionType {
        NONE(Optional.empty()),
        GZIP(Optional.of(".gz"));

        private final Optional<String> extension;

        private CompressionType(Optional<String> extension) {
            this.extension = Objects.requireNonNull(extension, "extension is null");
        }

        public Optional<String> getExtension() {
            return this.extension;
        }
    }
}

