package io.aiactiv.sdk.internal

import android.content.Context
import java.io.IOException

open class ValueMap() : MutableMap<String, Any?> {

    constructor(map: MutableMap<String, Any?>): this() {
        this.delegate = map
    }

    constructor(initialCapacity: Int): this() {
        this.delegate = LinkedHashMap(initialCapacity)
    }

    private var delegate: MutableMap<String, Any?> = LinkedHashMap()

    override val entries: MutableSet<MutableMap.MutableEntry<String, Any?>>
        get() = delegate.entries

    override val keys: MutableSet<String>
        get() = delegate.keys

    override val size: Int
        get() = delegate.size

    override val values: MutableCollection<Any?>
        get() = delegate.values


    override fun clear() {
        delegate.clear()
    }

    override fun containsKey(key: String): Boolean {
        return delegate.containsKey(key)
    }

    override fun containsValue(value: Any?): Boolean {
        return delegate.containsValue(value)
    }

    override fun get(key: String): Any? {
        return delegate[key]
    }

    override fun isEmpty(): Boolean {
        return delegate.isEmpty()
    }

    final override fun put(key: String, value: Any?): Any? {
        return delegate.put(key, value)
    }

    override fun putAll(from: Map<out String, Any?>) {
        return delegate.putAll(from)
    }

    override fun remove(key: String): Any? {
        return delegate.remove(key)
    }

    override fun equals(other: Any?): Boolean {
        return other == this || delegate == other
    }

    override fun hashCode(): Int {
        return delegate.hashCode()
    }

    override fun toString(): String {
        return delegate.toString()
    }

    open fun putValue(key: String, value: Any?): ValueMap {
        delegate[key] = value
        return this
    }

    fun getInt(key: String, defaultValue: Int): Int {
        val value = get(key)
        if (value is Int) {
            return value
        }
        if (value is Number) {
            return value.toInt()
        } else if (value is String) {
            try { return value.toInt() } catch (_: NumberFormatException) {}
        }
        return defaultValue
    }

    open fun getLong(key: String, defaultValue: Long): Long {
        val value = get(key)
        if (value is Long) {
            return value
        }
        if (value is Number) {
            return value.toLong()
        } else if (value is String) {
            try { return value.toLong() } catch (_: NumberFormatException) {}
        }
        return defaultValue
    }

    fun getDouble(key: String, defaultValue: Double): Double {
        val value = get(key)
        if (value is Double) { return value }
        if (value is Number) {
            return value.toDouble()
        } else if (value is String) {
            try { return value.toDouble() } catch (_: NumberFormatException) {}
        }
        return defaultValue
    }

    fun getChar(key: String, defaultValue: Char): Char {
        val value = get(key)
        if (value is Char) { return value }
        if (value != null && value is String) {
            if (value.length == 1) { return value[0] }
        }
        return defaultValue
    }

    fun getString(key: String): String? {
        val value = get(key)
        if (value is String) { return value }
        if (value != null) {
            return value.toString()
        }
        return null
    }

    fun getBoolean(key: String, defaultValue: Boolean): Boolean {
        val value = get(key)
        if (value is Boolean) {
            return value
        } else if (value is String) {
            return value.toBoolean()
        }
        return defaultValue
    }

    inline fun <reified T: Enum<T>> getEnum(enumType: Class<T>, key: String): T? {
        val value = get(key)
        if (value is T) { return value }
        if (value is String) {
            val stringValue = value.toString()
            return java.lang.Enum.valueOf(enumType, stringValue)
        }
        return null
    }

    @Suppress("UNCHECKED_CAST")
    fun getValueMap(key: Any): ValueMap? {
        val value = get(key)
        if (value is ValueMap) { return value }
        else if (value is MutableMap<*, *>) {
            return ValueMap(value as MutableMap<String, Any?>)
        }
        return null
    }

    fun <T: ValueMap> getValueMap(key: String, clazz: Class<T>): T? {
        val value = get(key)
        return coerceToValueMap(value, clazz)
    }

    private fun <T: ValueMap> coerceToValueMap(value: Any?, clazz: Class<T>): T? {
        return when {
            value == null -> null
            clazz.isAssignableFrom(value.javaClass) -> {
                @Suppress("UNCHECKED_CAST")
                return value as T
            }
            value is Map<*, *> -> createValueMap(value, clazz)
            else -> null
        }
    }

    fun <T: ValueMap> getList(key: Any, clazz: Class<T>): List<T>? {
        return when (val value = get(key)) {
            is List<*> -> {
                try {
                    val real = mutableListOf<T>()
                    value.forEach {
                        val typedValue = coerceToValueMap(it, clazz)
                        if (typedValue != null) {
                            real.add(typedValue)
                        }
                    }
                    return real
                } catch (e: Exception) {
                    return null
                }
            }
            else -> null
        }
    }

    fun toJsonObject() = Utils.toJsonObject(delegate)

    fun toStringMap(): Map<String, String> {
        val map = hashMapOf<String, String>()
        entries.forEach { map[it.key] = it.value as String }
        return map
    }

    companion object {
        fun <T> createValueMap(map: Map<*, *>, clazz: Class<T>): T {
            try {
                val constructor = clazz.getConstructor(Map::class.java)
                constructor.isAccessible = true
                return constructor.newInstance(map)
            } catch (e: Exception) {
                throw AssertionError("Could not create instance of ${clazz.canonicalName}.\n $e")
            }
        }
    }

    open class Cache<T: ValueMap>(
        context: Context,
        private val cartographer: Cartographer,
        private val key: String,
        tag: String,
        private val clazz: Class<T>,
    ) {
        private val sharedPreferences = Utils.getAIActivSharedPreferences(context, tag)

        private var value: T? = null

        fun get(): T? {
            when (value) {
                null -> {
                    val json = sharedPreferences.getString(key, null)
                    json?.let {
                        try {
                            val map = cartographer.fromJson(it)
                            value = create(map.toMutableMap())
                        } catch (e: IOException) {
                            return null
                        }
                    }
                }
            }
            return value
        }

        fun isSet() = sharedPreferences.contains(key)

        fun set(value: T) {
            this.value = value
            val json = cartographer.toJson(value)
            sharedPreferences.edit().putString(key, json).apply()
        }

        fun delete() = sharedPreferences.edit().remove(key).apply()

        open fun create(map: MutableMap<String, Any?>): T = createValueMap(map, clazz)
    }
}