package io.aiactiv.sdk.internal

import android.Manifest.permission
import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.content.Intent
import android.content.SharedPreferences
import android.content.pm.PackageInfo
import android.content.pm.PackageManager
import android.net.ConnectivityManager
import android.net.ParseException
import android.net.Uri
import android.os.Build
import org.json.JSONException
import org.json.JSONObject
import java.io.File
import java.io.IOException
import java.util.*

class Utils {
    companion object {

        const val THREAD_PREFIX = "Aiactiv-"
        const val DEFAULT_COLLECT_DEVICE_ID = true
        const val DEFAULT_FLUSH_QUEUE_SIZE = 20

        private val logger = Logger.with(Logger.LogLevel.DEBUG)

        private const val PERMISSION_CHECK_REPEAT_MAX_COUNT = 2

        /** Returns `date` formatted as yyyy-MM-ddThh:mm:ss.sssZ  */
        fun toISO8601String(date: Date): String {
            return ISO8601Utils.format(date)
        }

        fun parseISO8601Date(date: String): Date? {
            return ISO8601Utils.parse(date)
        }

        fun getAIActivSharedPreferences(context: Context, tag: String): SharedPreferences {
            return context.getSharedPreferences("aiactiv-sdk-$tag", Context.MODE_PRIVATE)
        }

        fun toJsonObject(map: Map<String, *>): JSONObject {
            val jsonObject = JSONObject()
            map.entries.forEach {
                val value = JSONObject.wrap(it.value)
                try {
                    jsonObject.put(it.key, value)
                } catch (e: JSONException) {
                    // Ignore values that JSONObject doesn't accept
                }
            }
            return jsonObject
        }

        fun hasPermission(context: Context, permission: String) =
            hasPermission(context, permission, 0)

        private fun hasPermission(context: Context, permission: String, repeatCount: Int): Boolean {
            return try {
                context.checkCallingOrSelfPermission(permission) == PackageManager.PERMISSION_GRANTED
            } catch (e: Exception) {
                logger.error(e, "Exception during permission check")
                repeatCount < PERMISSION_CHECK_REPEAT_MAX_COUNT && hasPermission(
                    context,
                    permission,
                    repeatCount + 1
                )
            }
        }

        /** Returns the system service for the given string.  */
        @Suppress("UNCHECKED_CAST")
        fun <T> getSystemService(context: Context, serviceConstant: String?): T {
            return context.getSystemService(serviceConstant!!) as T
        }

        fun getPackageInfo(context: Context): PackageInfo {
            val packageManager = context.packageManager
            try {
                return packageManager.getPackageInfo(context.packageName, 0)
            } catch (e: PackageManager.NameNotFoundException) {
                throw AssertionError("Package not found: ${context.packageName}")
            }
        }

        fun isConnected(context: Context): Boolean {
            if (hasPermission(context, permission.ACCESS_NETWORK_STATE).not()) {
                return true // Assume we have a connection and try to upload.
            }
            val cm: ConnectivityManager = getSystemService(context, Context.CONNECTIVITY_SERVICE)
            @SuppressLint("MissingPermission") val activeNetwork = cm.activeNetworkInfo
            return activeNetwork != null && activeNetwork.isConnectedOrConnecting
        }

        /** Get the string resource for the given key. Returns null if not found. */
        fun getResourceString(context: Context, name: String): String? {
            val id = getIdentifier(context, "string", name)
            return if (id != 0) context.getString(id) else null
        }

        /** Returns the referrer who started the Activity.  */
        fun getReferrer(activity: Activity): Uri? {
            return activity.referrer
        }

        /** Get the identifier for the resource with a given type and key. */
        private fun getIdentifier(context: Context, type: String, name: String): Int {
            return context.resources.getIdentifier(name, type, context.packageName)
        }

        /** Return [true] if a class with the given name is found. */
        fun isOnClassPath(className: String): Boolean {
            return try {
                Class.forName(className)
                true
            } catch (ignored: ClassNotFoundException) {
                false
            }
        }

        @Throws(IOException::class)
        fun createDirectory(location: File) {
            if ((location.exists() || location.mkdir() || location.isDirectory).not()) {
                throw IOException("Could not create directory at $location")
            }
        }
    }
}