package io.aiactiv.sdk.internal

import java.io.ByteArrayInputStream
import java.io.Closeable
import java.io.IOException
import java.io.InputStream
import java.util.LinkedList

internal abstract class PayloadQueue: Closeable {

    interface ElementVisitor {

        @Throws(IOException::class)
        fun read(inputStream: InputStream, length: Int): Boolean
    }

    abstract fun size(): Int

    @Throws(IOException::class)
    abstract fun remove(n: Int)

    @Throws(IOException::class)
    abstract fun add(data: ByteArray)

    @Throws(IOException::class)
    abstract fun forEach(visitor: ElementVisitor)

    @Throws(IOException::class)
    override fun close() {
        // No operation
    }

    companion object {

        class PersistenceQueue(private val queueFile: QueueFile): PayloadQueue() {
            override fun size(): Int = queueFile.size()

            @Throws(IOException::class)
            override fun remove(n: Int) {
                try {
                    queueFile.remove(n)
                } catch (e: ArrayIndexOutOfBoundsException) {
                    throw IOException(e)
                }
            }

            @Throws(IOException::class)
            override fun add(data: ByteArray) {
                queueFile.add(data)
            }

            @Throws(IOException::class)
            override fun forEach(visitor: ElementVisitor) {
                queueFile.forEach(visitor)
            }

            @Throws(IOException::class)
            override fun close() {
                queueFile.close()
            }
        }

        class MemoryQueue(): PayloadQueue() {

            var queue = LinkedList<ByteArray>()

            override fun size(): Int = queue.size

            @Throws(IOException::class)
            override fun remove(n: Int) {
                for (i in 0 until n) {
                    queue.remove()
                }
            }

            @Throws(IOException::class)
            override fun add(data: ByteArray) {
                queue.add(data)
            }

            @Throws(IOException::class)
            override fun forEach(visitor: ElementVisitor) {
                for (i in 0 until queue.size) {
                    val data = queue[i]
                    val shouldContinue = visitor.read(ByteArrayInputStream(data), data.size)
                    if (shouldContinue.not()) {
                        return
                    }
                }
            }
        }
    }
}