package io.aiactiv.sdk.internal

import android.util.Log

class Logger(
    private val tag: String,
    private val logLevel: LogLevel,
) {

    /** Log a verbose message. */
    fun verbose(format: String, vararg args: Any) {
        if (shouldLog(LogLevel.VERBOSE)) {
            Log.v(tag, String.format(format, args))
        }
    }

    /** Log a info message. */
    fun info(format: String, vararg args: Any) {
        if (shouldLog(LogLevel.INFO)) {
            Log.i(tag, String.format(format, args))
        }
    }

    /** Log a debug message. */
    fun debug(format: String, vararg args: Any) {
        if (shouldLog(LogLevel.DEBUG)) {
            Log.d(tag, String.format(format, args))
        }
    }

    /** Log an error message. */
    fun error(error: Throwable, format: String, vararg args: Any) {
        if (shouldLog(LogLevel.INFO)) {
            Log.e(tag, String.format(format, args), error)
        }
    }

    private fun shouldLog(level: LogLevel): Boolean { return  logLevel.ordinal >= level.ordinal }

    companion object {
        private const val DEFAULT_TAG = "Aiactiv";

        fun with(level: LogLevel): Logger { return Logger(DEFAULT_TAG, level) }
    }

    /** Controls the level of logging. */
    enum class LogLevel {
        /** No logging. */
        NONE,
        /** Log exceptions only. */
        INFO,
        /** Log exceptions and print debug output. */
        DEBUG,
        /** Same as {@link LogLevel#DEBUG}, and log transformations in bundled integrations. */
        VERBOSE;

        fun log(): Boolean { return this != NONE };
    }
}