package io.aiactiv.sdk.internal

import android.content.Context
import android.net.Uri
import android.util.Log
import androidx.annotation.WorkerThread
import com.google.gson.Gson
import io.aiactiv.sdk.SingletonHolder
import java.io.*
import java.net.HttpURLConnection
import java.net.URL
import java.util.zip.GZIPInputStream
import javax.net.ssl.HttpsURLConnection

class Client(context: Context) {

    private var userAgentGenerator = UserAgentGenerator(context)

    @WorkerThread
    inline fun <reified T> post(uri: Uri, requestHeaders: Map<String, String>, postData: Map<String, String>): ApiResponse<T> {
        val postDataBytes = convertPostDataToBytes(postData)
        var conn: HttpsURLConnection? = null
        try {
            conn = openPostConnection(uri, postDataBytes.size)
            setRequestHeaders(conn, requestHeaders)
            conn.connect()
            conn.outputStream.apply {
                write(postDataBytes)
                flush()
            }
            return getServiceResponse(conn)
        } catch (e: IOException) {
            return ApiResponse.createAsError(
                ApiResponseCode.NETWORK_ERROR,
                ApiError(e)
            )
        } finally {
            conn?.disconnect()
        }
    }

    @WorkerThread
    inline fun <reified T> postWithJson(uri: Uri,requestHeaders: Map<String, String>, postData: String): ApiResponse<T> {
        return sendRequestWithJson(HttpMethod.POST, uri, requestHeaders, postData)
    }

    @WorkerThread
    inline fun <reified T> putWithJson(uri: Uri, requestHeaders: Map<String, String>, postData: String): ApiResponse<T> {
        return sendRequestWithJson(
            HttpMethod.PUT,
            uri,
            requestHeaders,
            postData,
        )
    }

    @WorkerThread
    inline fun <reified T> sendRequestWithJson(method: HttpMethod, uri: Uri, requestHeaders: Map<String, String>, postData: String): ApiResponse<T> {
        Log.d(TAG, postData)
        val postDataBytes = postData.toByteArray()
        var conn: HttpsURLConnection? = null
        try {
            conn = openConnectionWithJson(uri, postDataBytes.size, method)
            setRequestHeaders(conn, requestHeaders)
            conn.connect()
            conn.outputStream.apply {
                write(postDataBytes)
                flush()
            }
            return getServiceResponse(conn)
        } catch (e: IOException) {
            return ApiResponse.createAsError(
                ApiResponseCode.NETWORK_ERROR,
                ApiError(e)
            )
        } finally {
            conn?.disconnect()
        }
    }

    @WorkerThread
    inline fun <reified T> get(uri: Uri, requestHeaders: Map<String, String>, queryParams: Map<String, String>): ApiResponse<T> {
        val fullUri = UriUtils.appendQueryParams(uri, queryParams)
        var conn: HttpsURLConnection? = null
        return try {
            conn = openGetConnection(fullUri)
            setRequestHeaders(conn, requestHeaders)
            conn.connect()
            getServiceResponse(conn)
        } catch (e: IOException) {
            ApiResponse.createAsError(
                ApiResponseCode.NETWORK_ERROR,
                ApiError(e)
            )
        } finally {
            conn?.disconnect()
        }
    }

    @WorkerThread
    inline fun <reified T> delete(uri: Uri, requestHeaders: Map<String, String>): ApiResponse<T> {
        var conn: HttpsURLConnection? = null
        return try {
            conn = openDeleteConnection(uri)
            setRequestHeaders(conn, requestHeaders)
            conn.connect()
            getServiceResponse(conn)
        } catch (e: IOException) {
            ApiResponse.createAsError(
                ApiResponseCode.NETWORK_ERROR,
                ApiError(e)
            )
        } finally {
            conn?.disconnect()
        }
    }

    @Throws(IOException::class)
    fun openPostConnection(uri: Uri, postDataSizeByte: Int
    ): HttpsURLConnection {
        val conn = openHttpsConnection(uri)
        conn.apply {
            instanceFollowRedirects = true
            connectTimeout = connectTimeoutMillis
            readTimeout = readTimeoutMillis
            requestMethod = HttpMethod.POST.name
            doOutput = true
            setRequestProperty("Accept-Encoding", "gzip")
            setRequestProperty("User-Agent", userAgentGenerator.userAgent)
            setRequestProperty("Content-Type", "application/x-www-form-urlencoded")
            setRequestProperty("Content-Length", postDataSizeByte.toString())
        }
        return conn
    }

    @Throws(IOException::class)
    fun openConnectionWithJson(uri: Uri, postDataSizeByte: Int, method: HttpMethod,
    ): HttpsURLConnection {
        val conn = openHttpsConnection(uri)
        conn.apply {
            instanceFollowRedirects = true
            connectTimeout = connectTimeoutMillis
            readTimeout = readTimeoutMillis
            requestMethod = method.name
            doOutput = true
            setRequestProperty("Accept-Encoding", "gzip")
            setRequestProperty("User-Agent", userAgentGenerator.userAgent)
            setRequestProperty("Content-Type", "application/json")
            setRequestProperty("Content-Length", postDataSizeByte.toString())
        }
        return conn
    }

    @Throws(IOException::class)
    fun openGetConnection(uri: Uri): HttpsURLConnection {
        val conn = openHttpsConnection(uri)
        conn.apply {
            instanceFollowRedirects = true
            connectTimeout = connectTimeoutMillis
            readTimeout = readTimeoutMillis
            requestMethod = HttpMethod.GET.name
            setRequestProperty("Accept-Encoding", "gzip")
            setRequestProperty("User-Agent", userAgentGenerator.userAgent)
        }
        return conn
    }

    @Throws(IOException::class)
    fun openDeleteConnection(uri: Uri): HttpsURLConnection {
        val conn = openHttpsConnection(uri)
        conn.apply {
            instanceFollowRedirects = true
            connectTimeout = connectTimeoutMillis
            readTimeout = readTimeoutMillis
            requestMethod = HttpMethod.DELETE.name
            setRequestProperty("Accept-Encoding", "gzip")
            setRequestProperty("User-Agent", userAgentGenerator.userAgent)
        }
        return conn
    }

    fun openHttpsConnection(uri: Uri): HttpsURLConnection {
        val urlConnection = URL(uri.toString()).openConnection()
        if (urlConnection !is HttpsURLConnection) {
            throw IllegalArgumentException("The scheme of the server url must be https.$uri")
        }
        return urlConnection
    }

    fun convertPostDataToBytes(postData: Map<String, String>): ByteArray {
        if (postData.isEmpty()) {
            return EMPTY_DATA
        }
        val uri = UriUtils.appendQueryParams("", postData)
        try {
            return uri.encodedQuery?.toByteArray(Charsets.UTF_8) ?: EMPTY_DATA
        } catch (e: UnsupportedEncodingException) {
            throw RuntimeException(e)
        }
    }

    fun setRequestHeaders(conn: HttpsURLConnection, requestHeaders: Map<String, String>
    ) {
        for (headerEntry in requestHeaders) {
            conn.setRequestProperty(headerEntry.key, headerEntry.value)
        }
    }

    @Throws(IOException::class)
    inline fun <reified T> getServiceResponse(conn: HttpsURLConnection): ApiResponse<T> {
        val responseCode = conn.responseCode
        try {
            Log.d(TAG, "Response Code $responseCode for ${conn.url.path}")
            if (responseCode == HttpURLConnection.HTTP_BAD_REQUEST) {
                val inputStream = getInputStreamFrom(conn)
                val reader = InputStreamReader(inputStream)
                val data = reader.readText()

                return ApiResponse.createAsError(
                    ApiResponseCode.INTERNAL_ERROR,
                    ApiError.createWithHttpResponseCode(
                        responseCode,
                        "Bad Request $data"
                    )
                )
            }

            if (responseCode == HttpURLConnection.HTTP_INTERNAL_ERROR) {
                return ApiResponse.createAsError(
                    ApiResponseCode.SERVER_ERROR,
                    ApiError.createWithHttpResponseCode(
                        responseCode,
                        "Server Internal Error"
                    )
                )
            }

            if (responseCode == HttpURLConnection.HTTP_NO_CONTENT) {
                return ApiResponse.createAsError(
                    ApiResponseCode.SERVER_ERROR,
                    ApiError.createWithHttpResponseCode(
                        responseCode,
                        "No Content"
                    )
                )
            }

            val inputStream = getInputStreamFrom(conn)
            val reader = InputStreamReader(inputStream)
            val data = reader.readText()
            val gson = Gson()
            return when (T::class) {
                String::class -> ApiResponse.createAsSuccess(data as T)
                else -> ApiResponse.createAsSuccess(gson.fromJson(data, T::class.java))
            }
        } catch (e: Exception) {
            return when(e) {
                is IOException -> ApiResponse.createAsError(
                    ApiResponseCode.NETWORK_ERROR,
                    ApiError(e, ApiError.ErrorCode.HTTP_RESPONSE_PARSE_ERROR)
                )
                else -> ApiResponse.createAsError(
                    ApiResponseCode.INTERNAL_ERROR,
                    ApiError("Internal Server Error")
                )
            }

        }
    }

    fun getInputStreamFrom(conn: HttpsURLConnection): InputStream {
        val responseCode = conn.responseCode
        val inputStream = if (responseCode < 400) conn.inputStream else conn.errorStream
        return if (isGzipUsed(conn)) GZIPInputStream(inputStream) else inputStream
    }

    private fun isGzipUsed(conn: HttpsURLConnection): Boolean {
        val contentEncodings = conn.headerFields["Content-Encoding"]
        contentEncodings?.let {
            for (contentEncoding in contentEncodings) {
                if (contentEncoding.equals("gzip", true)) {
                    return true
                }
            }
        }
        return false
    }

    private var connectTimeoutMillis = DEFAULT_CONNECT_TIMEOUT_MILLIS
    private var readTimeoutMillis = DEFAULT_READ_TIMEOUT_MILLIS

    enum class HttpMethod {
        POST, GET, DELETE, PUT
    }

    companion object: SingletonHolder<Client, Context>(::Client) {
        const val ADNETWORK_API_BASE_URI = "https://adnetwork-core-lb7.aiactiv.io"
        const val TAGS_API_BASE_URI = "https://tags.aiactiv.io/v1/b"
        private const val DEFAULT_CONNECT_TIMEOUT_MILLIS = 10 * 1000
        private const val DEFAULT_READ_TIMEOUT_MILLIS = 10 * 1000
        private val EMPTY_DATA = ByteArray(0)

        val TAG = Client::class.simpleName
    }
}
