package io.aiactiv.sdk.internal

import android.util.JsonReader
import android.util.JsonToken
import android.util.JsonWriter
import java.io.*

class Cartographer(
    private val isLenient: Boolean,
    private val prettyPrint: Boolean,
) {

    @Throws(IOException::class)
    fun fromJson(json: String): Map<String, Any?> {
        if (json.isEmpty()) {
            throw IllegalStateException("fromJson: json string is empty")
        }
        val stringReader = StringReader(json)
        stringReader.use {
            return fromJson(it)
        }
    }

    @Throws(IOException::class)
    fun fromJson(reader: Reader?): Map<String, Any?> {
        if (reader == null) {
            throw IllegalStateException("fromJson: reader is null")
        }
        val jsonReader = JsonReader(reader)
        jsonReader.isLenient = isLenient
        jsonReader.use {
            return readerToMap(it)
        }
    }

    @Throws(IOException::class)
    fun toJson(map: Map<*, *>): String {
        val stringWriter = StringWriter()
        stringWriter.use {
            toJson(map, it)
            return it.toString()
        }
    }

    @Throws(IOException::class)
    fun toJson(map: Map<*, *>, writer: Writer) {
        val jsonWriter = JsonWriter(writer)
        jsonWriter.isLenient = isLenient
        if (prettyPrint) {
            jsonWriter.setIndent("  ")
        }
        jsonWriter.use {
            return mapToWriter(map, it)
        }
    }

    companion object {

        val INSTANCE = Builder().lenient(true).prettyPrint(false).build()

        inline fun build(block: Builder.() -> Unit) = Builder().apply(block).build()

        @Throws(IOException::class)
        private fun readerToMap(reader: JsonReader): Map<String, Any?> {
            val map = LinkedHashMap<String, Any?>()
            reader.beginObject()
            while (reader.hasNext()) {
                map[reader.nextName()] = readValue(reader)
            }
            reader.endObject()
            return map
        }

        @Throws(IOException::class)
        private fun readerToList(reader: JsonReader): List<Any?> {
            val list = mutableListOf<Any?>()
            reader.beginArray()
            while (reader.hasNext()) {
                list.add(readValue(reader))
            }
            reader.endArray()
            return list
        }

        @Throws(IOException::class)
        private fun readValue(reader: JsonReader): Any? {
            return when (val token = reader.peek()) {
                JsonToken.BEGIN_OBJECT -> readerToMap(reader)
                JsonToken.BEGIN_ARRAY -> readerToList(reader)
                JsonToken.BOOLEAN -> reader.nextBoolean()
                JsonToken.NULL -> {
                    reader.nextNull()
                    return null
                }
                JsonToken.NUMBER -> reader.nextDouble()
                JsonToken.STRING -> reader.nextString()
                else -> {
                    throw IllegalStateException("Invalid Token $token")
                }
            }
        }

        @Throws(IOException::class)
        private fun writeValue(value: Any?, writer: JsonWriter) {
            when (value) {
                null -> {
                    writer.nullValue()
                }
                is Number -> {
                    var updated: Any? = value
                    if (value is Double && (value.isNaN() || value.isInfinite())) {
                        updated = 0.0
                    }
                    if (value is Float &&(value.isNaN() || value.isInfinite())) {
                        updated = 0.0
                    }
                    writer.value(updated as Number)
                }
                is Boolean -> {
                    writer.value(value)
                }
                is List<*> -> {
                    listToWriter(value, writer)
                }
                is Map<*, *> -> {
                    mapToWriter(value, writer)
                }
                is Array<*> -> {
                    arrayToWriter(value, writer)
                }
                else -> writer.value(value as String)
            }
        }

        @Throws(IOException::class)
        private fun listToWriter(list: List<*>, writer: JsonWriter) {
            writer.beginArray()
            list.forEach { writeValue(it, writer) }
            writer.endArray()
        }

        @Throws(IOException::class)
        private fun mapToWriter(map: Map<*, *>, writer: JsonWriter) {
            writer.beginObject()
            map.entries.forEach { entry ->
                writer.name(entry.key as String?)
                writeValue(entry.value, writer)
            }
            writer.endObject()
        }

        @Throws(IOException::class)
        private fun arrayToWriter(array: Array<*>, writer: JsonWriter) {
            writer.beginArray()
            array.forEach { writeValue(it, writer) }
            writer.endArray()
        }
    }

    class Builder {
        private var isLenient: Boolean = true
        private var prettyPrint: Boolean = false

        fun lenient(isLenient: Boolean): Builder {
            this.isLenient = isLenient
            return this
        }

        fun prettyPrint(prettyPrint: Boolean): Builder {
            this.prettyPrint = prettyPrint
            return this
        }

        fun build() = Cartographer(isLenient, prettyPrint)
    }
}