package io.aiactiv.sdk.internal

import android.content.Context
import android.util.Log
import com.google.gson.Gson
import io.aiactiv.sdk.adnetwork.api.BatchRequest
import io.aiactiv.sdk.adnetwork.models.batch.Batch
import io.aiactiv.sdk.adnetwork.models.batch.BatchContext
import io.aiactiv.sdk.adnetwork.models.batch.BatchType
import java.text.SimpleDateFormat
import java.util.*
import kotlin.collections.HashMap
import kotlin.concurrent.fixedRateTimer

class BatchController(
    private val context: Context,
    private val client: Client,
    private val gson: Gson,
) {
    private var batches = mutableListOf<Batch>()

    private var timer: Timer? = null

    private fun createBatch(type: BatchType, writeKey: String, data: Any? = null): Batch {
        val anonymousId = UUID.randomUUID().toString()
        val batchContext = BatchContext(context)
        val messageId = UUID.randomUUID().toString()
        val timeStamp = getCurrentISODatetime()
        return Batch(writeKey, anonymousId, messageId, "track", type.value, timeStamp).apply {
            batchContext.adData = data
            this.context = batchContext
        }
    }

    private fun getCurrentISODatetime(): String {
        val simpleDateFormat = SimpleDateFormat("yyyy-MM-dd'T'HH:mm'Z'", Locale.getDefault())
        simpleDateFormat.timeZone = TimeZone.getTimeZone("UTC")
        return simpleDateFormat.format(Date())
    }

    fun addBatch(type: BatchType, containerId: String, data: Any? = null) {
        val writeKey = "${containerId}@android"

        val batch = createBatch(type, writeKey, data)
        batches.add(batch)

        val sentAt = getCurrentISODatetime()
        val batchRequest = BatchRequest(writeKey, batches, sentAt)

        val body = gson.toJson(batchRequest)
        val size = body.toByteArray().size

        if (size >= MAX_BATCH_PACKAGE_SIZE) {
            sendBatches(body)
        } else {
            timer?.cancel()
            timer = fixedRateTimer("SendBatches", true, 5000, 60000) {
                sendBatches(body)
                timer?.cancel()
            }
        }
    }

    private fun sendBatches(body: String) {
        val uri = UriUtils.buildUri(Client.TAGS_API_BASE_URI)
        val headers = HashMap<String, String>()
        headers["Accept-Encoding"] = "gzip, deflate"
        val response = client.postWithJson<String>(uri, headers, body)
        response.responseData?.let {
            Log.d(TAG, "Send Batch result: $it")
        }
        batches.removeAll { it.timestamp < getCurrentISODatetime() }
    }

    companion object {
        private val TAG = BatchController::class.simpleName
        private const val MAX_BATCH_PACKAGE_SIZE = 500 * 1024
    }
}
