package io.aiactiv.sdk.internal

import android.os.Parcel
import android.os.Parcelable
import java.lang.Exception
import java.util.*
import kotlin.properties.Delegates

class ApiError() : Parcelable {

    enum class ErrorCode {
        ACTIVITY_NOT_FOUND,
        HTTP_RESPONSE_PARSE_ERROR,
        NOT_DEFINED,
    }

    private var responseCode by Delegates.notNull<Int>()
    private var message: String = ""
    private var errorCode: ErrorCode? = null

    constructor(parcel: Parcel) : this() {
        this.responseCode = parcel.readInt()
        this.message = parcel.readString().toString()
        val tempErrorCode = parcel.readInt()
        this.errorCode = if (tempErrorCode == -1) null else ErrorCode.values()[tempErrorCode]
    }

    constructor(e: Exception) : this(DEFAULT_HTTP_RESPONSE_CODE, e.toString(), ErrorCode.NOT_DEFINED)

    constructor(message: String) : this(DEFAULT_HTTP_RESPONSE_CODE, message, ErrorCode.NOT_DEFINED)

    constructor(e: Exception, errorCode: ErrorCode) : this(DEFAULT_HTTP_RESPONSE_CODE, e.toString(), errorCode)

    constructor(responseCode: Int, message: String, errorCode: ErrorCode) : this() {
        this.responseCode = responseCode
        this.message = message
        this.errorCode = errorCode
    }

    override fun writeToParcel(parcel: Parcel, flags: Int) {
        parcel.writeInt(responseCode)
        parcel.writeString(message)
        parcel.writeInt(errorCode?.ordinal ?: -1)
    }

    override fun describeContents(): Int {
        return 0
    }

    fun getMessage(): String {
        return this.message
    }

    override fun hashCode(): Int {
        return Objects.hash(responseCode, getMessage(), errorCode)
    }

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (other !is ApiError) return false
        return responseCode == other.responseCode &&
                Objects.equals(getMessage(), other.getMessage()) &&
                errorCode == other.errorCode
    }

    companion object {
        @JvmField
        val CREATOR: Parcelable.Creator<ApiError> = object : Parcelable.Creator<ApiError> {
            override fun createFromParcel(parcel: Parcel): ApiError {
                return ApiError(parcel)
            }

            override fun newArray(size: Int): Array<ApiError?> {
                return Array(size) { ApiError() }
            }
        }

        private const val DEFAULT_HTTP_RESPONSE_CODE = -1

        val DEFAULT = ApiError(DEFAULT_HTTP_RESPONSE_CODE, "Not Defined Error", ErrorCode.NOT_DEFINED)

        fun createWithHttpResponseCode(responseCode: Int, errorString: String): ApiError {
            return ApiError(responseCode, errorString, ErrorCode.NOT_DEFINED)
        }

        fun createWithHttpResponseCode(responseCode: Int, e: Exception): ApiError {
            return createWithHttpResponseCode(responseCode, e.toString())
        }
    }
}