package io.aiactiv.sdk.analytics.integrations

import java.util.*

class ScreenPayload(
    messageId: String?,
    timestamp: Date?,
    context: Map<String, Any?>?,
    integrations: Map<String, Any?>?,
    userId: String?,
    anonymousId: String?,
    name: String?,
    category: String?,
    properties: Map<String, Any?>?,
    nanosecondTimestamps: Boolean
): BasePayload(
    Type.screen,
    messageId,
    timestamp,
    context,
    integrations,
    userId,
    anonymousId,
    nanosecondTimestamps
) {

    init {
        if (name.isNullOrEmpty().not()) put(NAME_KEY, name)
        if (category.isNullOrEmpty().not()) put(CATEGORY_KEY, category)
        put(PROPERTIES_KEY, properties)
    }

    fun name() = getString(NAME_KEY)
    fun category() = getString(CATEGORY_KEY)
    fun properties() = getValueMap(PROPERTIES_KEY, io.aiactiv.sdk.Properties::class.java)

    companion object {
        const val CATEGORY_KEY = "category"
        const val NAME_KEY = "name"
        const val PROPERTIES_KEY = "properties"
    }

    override fun toBuilder(): Builder {
        return Builder(this)
    }

    class Builder(): BasePayload.Builder<ScreenPayload, Builder>() {

        private var name: String? = null
        private var category: String? = null
        private var properties: Map<String, Any?>? = null

        constructor(payload: ScreenPayload): this() {
            this.name = payload.name()
            this.category = payload.category()
            this.properties = payload.properties()
        }

        fun name(name: String?): Builder {
            this.name = name
            return this
        }

        fun category(category: String?): Builder {
            this.category = category
            return this
        }

        fun properties(properties: MutableMap<String, Any?>): Builder {
            this.properties = properties
            return this
        }

        override fun realBuild(
            messageId: String?,
            timestamp: Date?,
            context: Map<String, Any?>?,
            integrations: Map<String, Any?>?,
            userId: String?,
            anonymousId: String?,
            nanosecondTimestamps: Boolean
        ): ScreenPayload {
            if (name.isNullOrEmpty() && category.isNullOrEmpty()) {
                throw NullPointerException("either name or category is required")
            }

            val properties = this.properties ?: emptyMap()
            return ScreenPayload(
                messageId,
                timestamp,
                context,
                integrations,
                userId,
                anonymousId,
                name,
                category,
                properties,
                nanosecondTimestamps
            )
        }

        override fun self(): Builder {
            return this
        }

    }
}