package io.aiactiv.sdk.analytics.integrations

import io.aiactiv.sdk.analytics.AnalyticsContext
import io.aiactiv.sdk.internal.ValueMap
import io.aiactiv.sdk.internal.ISO8601Utils
import io.aiactiv.sdk.internal.NanoDate
import java.util.*
import kotlin.collections.LinkedHashMap

abstract class BasePayload() : ValueMap() {

    constructor(
        type: Type,
        messageId: String?,
        timestamp: Date?,
        context: Map<String, Any?>?,
        integrations: Map<String, Any?>?,
        userId: String?,
        anonymousId: String?,
        nanosecondTimestamps: Boolean,
    ) : this() {

        put(CHANNEL_KEY, Channel.mobile.name)
        put(TYPE_KEY, type.name)
        put(MESSAGE_ID, messageId)

        timestamp?.let {
            if (nanosecondTimestamps) {
                put(TIMESTAMP_KEY, ISO8601Utils.formatNanos(timestamp))
            } else {
                put(TIMESTAMP_KEY, ISO8601Utils.format(timestamp))
            }
        }

        put(CONTEXT_KEY, context)
        put(INTEGRATIONS_KEY, integrations)
        put(USER_ID_KEY, userId)
        put(ANONYMOUS_ID_KEY, anonymousId)
    }

    enum class Type {
        alias,
        group,
        identity,
        screen,
        track
    }

    enum class Channel {
        // browser,
        mobile,
        // server
    }

    /** The type of message.  */
    open fun type(): Type? {
        return getEnum(Type::class.java, TYPE_KEY)
    }

    /**
     * The user ID is an identifier that unique identifies the user in your database. Ideally it
     * should not be an email address, because emails can change, whereas a database ID can't.
     */
    open fun userId(): String? {
        return getString(USER_ID_KEY)
    }

    /**
     * The anonymous ID is an identifier that uniquely (or close enough) identifies the user, but
     * isn't from your database. This is useful in cases where you are able to uniquely identifier
     * the user between visits before they sign up thanks to a cookie, or session ID or device ID.
     * In our mobile and browser libraries we will automatically handle sending the anonymous ID.
     */
    open fun anonymousId(): String? {
        return getString(ANONYMOUS_ID_KEY)
    }

    /** A randomly generated unique id for this message.  */
    open fun messageId(): String? {
        return getString(MESSAGE_ID)
    }

    /**
     * Set a timestamp the event occurred.
     *
     *
     * This library will automatically create and attach a timestamp to all events.
     *
     */
    open fun timestamp(): Date? {
        // It's unclear if this will ever be null. So we're being safe.
        val timestamp = getString(TIMESTAMP_KEY)
        if (timestamp.isNullOrEmpty()) {
            return null
        }
        return if (timestamp.length == "yyyy-MM-ddThh:mm:ss.fffZ".length) {
            ISO8601Utils.parse(timestamp)
        } else {
            ISO8601Utils.parseWithNanos(timestamp)
        }
    }

    /**
     * A dictionary of integration names that the message should be proxied to. 'All' is a special
     * name that applies when no key for a specific integration is found, and is case-insensitive.
     */
    open fun integrations(): ValueMap? {
        return getValueMap(INTEGRATIONS_KEY)
    }

    /**
     * The context is a dictionary of extra information that provides useful context about a
     * message, for example ip address or locale.
     *
     * @see [Context fields](https://segment.com/docs/spec/common/.context)
     */
    internal fun context(): AnalyticsContext? {
        return getValueMap(CONTEXT_KEY, AnalyticsContext::class.java)
    }

    override fun putValue(key: String, value: Any?): BasePayload {
        super.putValue(key, value)
        return this
    }

    abstract fun toBuilder(): Builder<*, *>

    companion object {
        const val TYPE_KEY = "type"
        const val ANONYMOUS_ID_KEY = "anonymousId"
        const val CHANNEL_KEY = "channel"
        const val MESSAGE_ID = "messageId"
        const val CONTEXT_KEY = "context"
        const val INTEGRATIONS_KEY = "integrations"
        const val TIMESTAMP_KEY = "timestamp"
        const val USER_ID_KEY = "userId"
    }

    abstract class Builder<P : BasePayload, B : Builder<P, B>>() {

        private var messageId: String? = null
        private var timestamp: Date? = null
        private var context: Map<String, Any?>? = null
        private var integrationsBuilder: MutableMap<String, Any?>? = null
        private var userId: String? = null
        private var anonymousId: String? = null
        private var nanosecondTimestamps: Boolean = false

        constructor(payload: P) : this() {
            val timestampString = payload.getString(TIMESTAMP_KEY)
            nanosecondTimestamps = timestampString != null && timestampString.length > 24
            messageId = payload.messageId()
            timestamp = payload.timestamp()
            context = payload.context()
            integrationsBuilder = payload.integrations()
            userId = payload.userId()
            anonymousId = payload.anonymousId()
        }

        fun messageId(messageId: String): B {
            this.messageId = messageId
            return self()
        }

        fun timestamp(timestamp: Date): B {
            this.timestamp = timestamp
            return self()
        }

        fun context(context: Map<String, Any?>): B {
            this.context = context
            return self()
        }

        fun integrations(key: String, enable: Boolean): B {
            if (integrationsBuilder == null) {
                integrationsBuilder = LinkedHashMap()
            }
            integrationsBuilder?.put(key, enable)
            return self()
        }

        fun integrations(key: String, options: Map<String, Any?>): B {
            if (integrationsBuilder == null) {
                integrationsBuilder = LinkedHashMap()
            }
            integrationsBuilder?.put(key, options)
            return self()
        }

        fun integrations(options: Map<String, Any?>): B {
            if (integrationsBuilder == null) {
                integrationsBuilder = LinkedHashMap()
            }
            integrationsBuilder?.putAll(options)
            return self()
        }

        fun userId(userId: String?): B {
            this.userId = userId
            return self()
        }

        fun anonymousId(anonymousId: String?): B {
            this.anonymousId = anonymousId
            return self()
        }

        fun nanosecondTimestamps(enable: Boolean): B {
            this.nanosecondTimestamps = enable
            return self()
        }

        fun isUserIdSet() = !userId.isNullOrEmpty()

        abstract fun realBuild(
            messageId: String?,
            timestamp: Date?,
            context: Map<String, Any?>?,
            integrations: Map<String, Any?>?,
            userId: String?,
            anonymousId: String?,
            nanosecondTimestamps: Boolean,
        ): P

        abstract fun self(): B

        fun build(): P {
            if (timestamp == null) {
                timestamp = if (nanosecondTimestamps) NanoDate() else Date()
            }
            return realBuild(
                messageId,
                timestamp,
                context,
                integrationsBuilder,
                userId,
                anonymousId,
                nanosecondTimestamps,
            )
        }
    }
}