package io.aiactiv.sdk.analytics

import android.content.Context
import io.aiactiv.sdk.internal.Cartographer
import io.aiactiv.sdk.internal.Utils
import io.aiactiv.sdk.internal.ValueMap
import java.text.ParseException
import java.util.*

class Traits: ValueMap {

    constructor(): super()
    constructor(delegate: MutableMap<String, Any?>): super(delegate)
    constructor(initialCapacity: Int): super(initialCapacity)

    override fun putValue(key: String, value: Any?): Traits {
        super.putValue(key, value)
        return this
    }

    fun unmodifiableCopy(): Traits {
        val map = LinkedHashMap<String, Any?>(this)
        return Traits(Collections.unmodifiableMap(map))
    }

    fun putAnonymousId(id: String) = this.putValue(ANONYMOUS_ID_KEY, id)
    fun anonymousId() = getString(ANONYMOUS_ID_KEY)

    fun putUserId(id: String) = putValue(USER_ID_KEY, id)
    fun userId() = getString(USER_ID_KEY)

    fun currentId(): String? {
        val userId = userId()
        return if (userId.isNullOrEmpty()) anonymousId() else userId
    }

    fun putAddress(address: Address) = putValue(ADDRESS_KEY, address)
    fun address(): Address? = getValueMap(ADDRESS_KEY, Address::class.java)

    fun putAge(age: Int) = putValue(AGE_KEY, age)
    fun age(): Int = getInt(AGE_KEY, 0)

    fun putAvatar(avatar: String) = putValue(AVATAR_KEY, avatar)
    fun avatar(): String? = getString(AVATAR_KEY)

    fun putBirthday(birthday: Date) = putValue(BIRTHDAY_KEY, Utils.toISO8601String(birthday))
    fun birthday(): Date? {
        try {
            val birthday = getString(BIRTHDAY_KEY)
            if (birthday.isNullOrEmpty()) return null
            return Utils.parseISO8601Date(birthday)
        } catch (e: ParseException) {
            return null
        }
    }

    fun putCreateAt(createAt: String) = putValue(CREATED_AT_KEY, createAt)
    fun createAt(): String? = getString(CREATED_AT_KEY)

    fun putDescription(description: String) = putValue(DESCRIPTION_KEY, description)
    fun description(): String? = getString(DESCRIPTION_KEY)

    fun putEmail(email: String) = putValue(EMAIL_KEY, email)
    fun email(): String? = getString(EMAIL_KEY)

    fun putEmployees(employees: Int) = putValue(EMPLOYEES_KEY, employees)
    fun employees(): Int = getInt(EMPLOYEES_KEY, 0)

    fun putFirstName(firstName: String) = putValue(FIRST_NAME_KEY, firstName)
    fun firstName(): String? = getString(FIRST_NAME_KEY)

    fun putGender(gender: String) = putValue(GENDER_KEY, gender)
    fun gender(): String? = getString(GENDER_KEY)

    fun putIndustry(industry: String) = putValue(INDUSTRY_KEY, industry)
    fun industry(): String? = getString(INDUSTRY_KEY)

    fun putLastName(lastName: String) = putValue(LAST_NAME_KEY, lastName)
    fun lastName(): String? = getString(LAST_NAME_KEY)

    fun putName(name: String) = putValue(NAME_KEY, name)
    fun name(): String? {
        val name = getString(NAME_KEY)
        if (name.isNullOrEmpty() && firstName().isNullOrEmpty() && lastName().isNullOrEmpty()) {
            return null
        }
        if (name.isNullOrEmpty()) {
            val builder = StringBuilder()
            val firstName = firstName()
            var appendSpace = false
            if (firstName.isNullOrEmpty()) {
                appendSpace = true;
                builder.append(firstName)
            }

            val lastName = lastName()
            if (lastName.isNullOrEmpty()) {
                if (appendSpace) builder.append(" ")
                builder.append(lastName)
            }

            return builder.toString()
        }
        return name
    }

    fun putPhone(phone: String) = putValue(PHONE_KEY, phone)
    fun phone(): String? = getString(PHONE_KEY)

    fun putTitle(title: String) = putValue(TITLE_KEY, title)
    fun title(): String? = getString(TITLE_KEY)

    fun putUsername(username: String) = putValue(USERNAME_KEY, username)
    fun username(): String? = getString(USERNAME_KEY)

    fun putWebsite(website: String) = putValue(WEBSITE_KEY, website)
    fun website(): String? = getString(WEBSITE_KEY)

    companion object {
        private const val AVATAR_KEY = "avatar"
        private const val CREATED_AT_KEY = "createdAt"
        private const val DESCRIPTION_KEY = "description"
        private const val EMAIL_KEY = "email"
        private const val ANONYMOUS_ID_KEY = "anonymousId"
        private const val USER_ID_KEY = "userId"
        private const val NAME_KEY = "name"
        private const val PHONE_KEY = "phone"
        private const val WEBSITE_KEY = "website"

        // For Identify Calls
        private const val AGE_KEY = "age"
        private const val BIRTHDAY_KEY = "birthday"
        private const val FIRST_NAME_KEY = "firstName"
        private const val GENDER_KEY = "gender"
        private const val LAST_NAME_KEY = "lastName"
        private const val TITLE_KEY = "title"
        private const val USERNAME_KEY = "username"

        // For Group calls
        private const val EMPLOYEES_KEY = "employees"
        private const val INDUSTRY_KEY = "industry"

        // Address
        private const val ADDRESS_KEY = "address"

        fun create(): Traits {
            return Traits(mutableMapOf()).apply {
                putAnonymousId(UUID.randomUUID().toString())
            }
        }
    }

    class Address: ValueMap {

        constructor(): super()
        constructor(delegate: MutableMap<String, Any?>): super(delegate)

        override fun putValue(key: String, value: Any?): Address {
            super.putValue(key, value)
            return this
        }

        fun putCity(city: String) = putValue(ADDRESS_CITY_KEY, city)
        fun city(): String? = getString(ADDRESS_CITY_KEY)

        fun putCountry(country: String) = putValue(ADDRESS_COUNTRY_KEY, country)
        fun country(): String? = getString(ADDRESS_COUNTRY_KEY)

        fun putPostalCode(postalCode: String) = putValue(ADDRESS_POSTAL_CODE_KEY, postalCode)
        fun postalCode(): String? = getString(ADDRESS_POSTAL_CODE_KEY)

        fun putState(state: String) = putValue(ADDRESS_STATE_KEY, state)
        fun state(): String? = getString(ADDRESS_STATE_KEY)

        fun putStreet(street: String) = putValue(ADDRESS_STREET_KEY, street)
        fun street(): String? = getString(ADDRESS_STREET_KEY)

        companion object {
            private const val ADDRESS_CITY_KEY = "city"
            private const val ADDRESS_COUNTRY_KEY = "country"
            private const val ADDRESS_POSTAL_CODE_KEY = "postalCode"
            private const val ADDRESS_STATE_KEY = "state"
            private const val ADDRESS_STREET_KEY = "street"
        }
    }

    class Cache(
        context: Context,
        cartographer: Cartographer,
        tag: String
    ): ValueMap.Cache<Traits>(
        context, cartographer, TRAITS_CACHE_PREFIX + tag, tag, Traits::class.java
    ) {

        override fun create(map: MutableMap<String, Any?>): Traits {
            return Traits(map)
        }

        companion object {
            private const val TRAITS_CACHE_PREFIX = "traits-"
        }
    }
}