package io.aiactiv.sdk.analytics

import android.Manifest.permission
import android.content.Context
import android.content.pm.PackageManager
import android.net.ConnectivityManager
import android.os.Build
import android.telephony.TelephonyManager
import android.util.DisplayMetrics
import android.view.WindowManager
import io.aiactiv.sdk.BuildConfig
import io.aiactiv.sdk.internal.Logger
import io.aiactiv.sdk.internal.Utils
import io.aiactiv.sdk.internal.ValueMap
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.util.*
import java.util.concurrent.ConcurrentHashMap

internal class AnalyticsContext: ValueMap {

    constructor(): super()
    constructor(delegate: MutableMap<String, Any?>): super(delegate)
    constructor(initialCapacity: Int): super(initialCapacity)

    fun attachAdvertisingId(context: Context, logger: Logger, scope: CoroutineScope) {
        if (Utils.isOnClassPath("com.google.android.gms.ads.identifier.AdvertisingIdClient")) {
            scope.launch {
                val device = getValueMap(DEVICE_KEY, Device::class.java) ?: return@launch

                val advertisingInfo = Class.forName("com.google.android.gms.ads.identifier.AdvertisingIdClient")
                    .getMethod("getAdvertisingIdInfo", Context::class.java)
                    .invoke(null, context)
                val isLimitAdTrackingEnabled = advertisingInfo.javaClass
                    .getMethod("isLimitAdTrackingEnabled")
                    .invoke(advertisingInfo) as Boolean

                if (isLimitAdTrackingEnabled) {
                    logger.debug("Not collecting advertising ID because isLimitAdTrackingEnabled (Google Play Services) is true.")
                    device.putAdvertisingInfo(null, false)
                }

                val advertisingId = advertisingInfo.javaClass.getMethod("getId").invoke(advertisingInfo) as String?
                device.putAdvertisingInfo(advertisingId, true)
                logger.debug("Collect advertisingId $advertisingId")
            }
        } else {
            logger.debug("Not collecting advertising ID because " +
                    "com.google.android.gms.ads.identifier.AdvertisingIdClient " +
                    "was not found on the classpath.")
        }
    }

    /**
     * Fill this instance with application info from the provided [Context]. No need to expose
     * a getter for this for bundled integrations (they'll automatically fill what they need
     * themselves).
     */
    fun putApp(context: Context) {
        try {
            val packageManager = context.packageManager
            val packageInfo = packageManager.getPackageInfo(context.packageName, 0)
            val app: MutableMap<String, Any?> = mutableMapOf()
            putUndefinedIfNull(
                app, APP_NAME_KEY, packageInfo.applicationInfo.loadLabel(packageManager)
            )
            putUndefinedIfNull(app, APP_VERSION_KEY, packageInfo.versionName)
            putUndefinedIfNull(app, APP_NAMESPACE_KEY, packageInfo.packageName)
            app[APP_BUILD_KEY] = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
                packageInfo.longVersionCode.toString()
            } else {
                packageInfo.versionCode.toString()
            }
            put(APP_KEY, app)
        } catch (e: PackageManager.NameNotFoundException) {
            // ignore
        }
    }

    /** Fill this instance with device info from the provided [Context].  */
    fun putDevice(collectDeviceID: Boolean) {
        val device = Device()

        // use empty string to indicate device id is not yet ready.
        // the device id will be populated async (see `attachDeviceId`)
        val identifier = if (collectDeviceID) "" else traits()?.anonymousId()
        device[Device.DEVICE_ID_KEY] = identifier
        device[Device.DEVICE_MANUFACTURER_KEY] = Build.MANUFACTURER
        device[Device.DEVICE_MODEL_KEY] = Build.MODEL
        device[Device.DEVICE_NAME_KEY] = Build.DEVICE
        device[Device.DEVICE_TYPE_KEY] = "android"
        put(DEVICE_KEY, device)
    }

    /** Fill this instance with library information.  */
    fun putLibrary() {
        val library = mutableMapOf<String, String>()
        library[LIBRARY_NAME_KEY] = "aiactiv-360sdk"
        library[LIBRARY_VERSION_KEY] = BuildConfig.VERSION_NAME
        put(LIBRARY_KEY, library)
    }

    /**
     * Fill this instance with network information. No need to expose a getter for this for bundled
     * integrations (they'll automatically fill what they need themselves).
     */
    fun putNetwork(context: Context) {
        val network = mutableMapOf<String, Any>()
        if (Utils.hasPermission(context, permission.ACCESS_NETWORK_STATE)) {
            val connectivityManager: ConnectivityManager =
                Utils.getSystemService(context, Context.CONNECTIVITY_SERVICE)

            val wifiInfo = connectivityManager.getNetworkInfo(ConnectivityManager.TYPE_WIFI)
            network[NETWORK_WIFI_KEY] = if (wifiInfo != null && wifiInfo.isConnected) {
                wifiInfo.isConnected.toString()
            } else "false"
            val bluetoothInfo =
                connectivityManager.getNetworkInfo(ConnectivityManager.TYPE_BLUETOOTH)
            network[NETWORK_BLUETOOTH_KEY] = if (bluetoothInfo != null && bluetoothInfo.isConnected) {
                bluetoothInfo.isConnected.toString()
            } else "false"
            val cellularInfo =
                connectivityManager.getNetworkInfo(ConnectivityManager.TYPE_MOBILE)
            network[NETWORK_CELLULAR_KEY] = if (cellularInfo != null && cellularInfo.isConnected) {
                cellularInfo.isConnected.toString()
            } else "false"
        }
        val telephonyManager: TelephonyManager =
            Utils.getSystemService(context, Context.TELEPHONY_SERVICE)
        network[NETWORK_CARRIER_KEY] = telephonyManager.networkOperatorName
        put(NETWORK_KEY, network)
    }

    /** Fill this instance with operating system information.  */
    fun putOs() {
        val os = mutableMapOf<String, String>()
        os[OS_NAME_KEY] = "Android"
        os[OS_VERSION_KEY] = Build.VERSION.RELEASE
        put(OS_KEY, os)
    }

    /**
     * Fill this instance with application info from the provided [Context]. No need to expose
     * a getter for this for bundled integrations (they'll automatically fill what they need
     * themselves).
     */
    fun putScreen(context: Context) {
        val screen = mutableMapOf<String, Any>()
        val manager: WindowManager = Utils.getSystemService(context, Context.WINDOW_SERVICE)

        val displayMetrics = DisplayMetrics()
        manager.defaultDisplay?.getMetrics(displayMetrics)
        screen[SCREEN_DENSITY_KEY] = displayMetrics.density.toString()
        screen[SCREEN_HEIGHT_KEY] = displayMetrics.heightPixels.toString()
        screen[SCREEN_WIDTH_KEY] = displayMetrics.widthPixels.toString()
        // put(SCREEN_KEY, screen)
    }

    /**
     * Return the [Traits] attached to this instance.
     */
    fun traits(): Traits? {
        return getValueMap(TRAITS_KEY, Traits::class.java)
    }

    fun setTraits(traits: Traits) {
        put(TRAITS_KEY, traits.unmodifiableCopy())
    }

    companion object {
        private const val LOCALE_KEY = "locale"
        private const val TRAITS_KEY = "traits"
        private const val USER_AGENT_KEY = "userAgent"
        private const val TIMEZONE_KEY = "timezone"

        // App
        private const val APP_KEY = "app"
        private const val APP_NAME_KEY = "name"
        private const val APP_VERSION_KEY = "version"
        private const val APP_NAMESPACE_KEY = "namespace"
        private const val APP_BUILD_KEY = "build"

        // Campaign
        private const val CAMPAIGN_KEY = "campaign"

        // Device
        const val DEVICE_KEY = "device"

        // Library
        private const val LIBRARY_KEY = "library"
        private const val LIBRARY_NAME_KEY = "name"
        private const val LIBRARY_VERSION_KEY = "version"

        // Location
        private const val LOCATION_KEY = "location"

        // Network
        private const val NETWORK_KEY = "network"
        private const val NETWORK_BLUETOOTH_KEY = "bluetooth"
        private const val NETWORK_CARRIER_KEY = "carrier"
        private const val NETWORK_CELLULAR_KEY = "cellular"
        private const val NETWORK_WIFI_KEY = "wifi"

        // OS
        private const val OS_KEY = "os"
        private const val OS_NAME_KEY = "name"
        private const val OS_VERSION_KEY = "version"

        // Referrer
        private const val REFERRER_KEY = "referrer"

        // Screen
        private const val SCREEN_KEY = "screen"
        private const val SCREEN_DENSITY_KEY = "density"
        private const val SCREEN_HEIGHT_KEY = "height"
        private const val SCREEN_WIDTH_KEY = "width"

        @Synchronized
        fun create(context: Context, traits: Traits, collectDeviceId: Boolean): AnalyticsContext {
            val analyticsContext = AnalyticsContext(ConcurrentHashMap<String, Any?>())
            analyticsContext.putApp(context)
            analyticsContext.setTraits(traits)
            analyticsContext.putDevice(collectDeviceId)
            analyticsContext.putLibrary()
            analyticsContext[LOCALE_KEY] =
                Locale.getDefault().language + "-" + Locale.getDefault().country
            analyticsContext.putNetwork(context)
            analyticsContext.putOs()
            analyticsContext.putScreen(context)
            putUndefinedIfNull(
                analyticsContext,
                USER_AGENT_KEY,
                System.getProperty("http.agent")
            )
            putUndefinedIfNull(
                analyticsContext,
                TIMEZONE_KEY,
                TimeZone.getDefault().id
            )
            return analyticsContext
        }

        fun putUndefinedIfNull(target: MutableMap<String, Any?>, key: String, value: CharSequence?) {
            if (value.isNullOrEmpty()) {
                target[key] = "undefined"
            } else {
                target[key] = value
            }
        }
    }

    /** Information about the device.  */
    class Device : ValueMap {

        constructor(): super()
        constructor(delegate: MutableMap<String, Any?>): super(delegate)
        constructor(initialCapacity: Int): super(initialCapacity)

        override fun putValue(key: String, value: Any?): Device {
            super.putValue(key, value)
            return this
        }

        /** Set the advertising information for this device.  */
        fun putAdvertisingInfo(advertisingId: String?, adTrackingEnabled: Boolean) {
            if (adTrackingEnabled && !advertisingId.isNullOrEmpty()) {
                put(DEVICE_ADVERTISING_ID_KEY, advertisingId)
            }
            put(DEVICE_AD_TRACKING_ENABLED_KEY, adTrackingEnabled)
        }

        /** Set a device token.  */
        fun putDeviceToken(token: String?): Device {
            return putValue(DEVICE_TOKEN_KEY, token)
        }

        companion object {
            const val DEVICE_ID_KEY = "id"
            const val DEVICE_MANUFACTURER_KEY = "manufacturer"
            const val DEVICE_MODEL_KEY = "model"
            const val DEVICE_NAME_KEY = "name"
            const val DEVICE_TYPE_KEY = "type"
            const val DEVICE_TOKEN_KEY = "token"
            const val DEVICE_ADVERTISING_ID_KEY = "advertisingId"
            const val DEVICE_AD_TRACKING_ENABLED_KEY = "adTrackingEnabled"
        }
    }
}