package io.aiactiv.sdk.analytics

import android.content.Context
import android.os.Handler
import android.os.HandlerThread
import android.os.Looper
import android.os.Message
import io.aiactiv.sdk.Analytics
import io.aiactiv.sdk.analytics.integrations.*
import io.aiactiv.sdk.analytics.integrations.Integration
import io.aiactiv.sdk.internal.*
import io.aiactiv.sdk.internal.PayloadQueue
import java.io.*
import java.util.zip.GZIPOutputStream
import javax.net.ssl.HttpsURLConnection

internal class AiactivIntegration(
    private val context: Context,
    private val writeKey: String,
    private val bundleIntegrations: Map<String, Boolean>,
    private val payloadQueue: PayloadQueue,
    private val logger: Logger,
    private val cartographer: Cartographer,
): Integration<Void>() {

    private val aiactivThread = HandlerThread(AIACTIV_THREAD_NAME, android.os.Process.THREAD_PRIORITY_BACKGROUND)
    private val handler: AiactivDispatcherHandler

    private val client = Client.getInstance(context)

    private val flushLock = Any()

    init {
        aiactivThread.start()
        handler = AiactivDispatcherHandler(aiactivThread.looper, this)
    }

    override fun track(track: TrackPayload) = dispatchQueue(track)

    override fun identify(identify: IdentifyPayload) = dispatchQueue(identify)

    override fun screen(screen: ScreenPayload) = dispatchQueue(screen)

    private fun dispatchQueue(payload: BasePayload) {
        handler.sendMessage(handler.obtainMessage(AiactivDispatcherHandler.REQUEST_ENQUEUE, payload))
    }

    private fun performEnqueue(original: BasePayload) {
        val providedIntegrations = original.integrations()
        val combinedIntegrations = LinkedHashMap<String, Any?>((providedIntegrations?.size ?: 0) + bundleIntegrations.size)
        if (providedIntegrations != null) {
            combinedIntegrations.putAll(providedIntegrations)
        }
        combinedIntegrations.putAll(bundleIntegrations)
        combinedIntegrations.remove(AIACTIV_KEY)
        val payload = ValueMap()
        payload.putAll(original)
        payload["integrations"] = combinedIntegrations

        if (payloadQueue.size() >= MAX_QUEUE_SIZE) {
            synchronized(flushLock) {
                if (payloadQueue.size() >= MAX_QUEUE_SIZE) {
                    logger.info("Queue is at max capacity ${payloadQueue.size()}, removing oldest payload")
                    try {
                        payloadQueue.remove(1)
                    } catch (e: IOException) {
                        logger.error(e, "Unable to remove oldest payload from queue.")
                        return
                    }
                }
            }
        }

        try {
            val bos = ByteArrayOutputStream()
            cartographer.toJson(payload, OutputStreamWriter(bos))
            val bytes = bos.toByteArray()
            if (bytes.isEmpty() || bytes.size > MAX_QUEUE_SIZE) {
                throw IOException("Could not serialize payload $payload")
            }
            payloadQueue.add(bytes)
        } catch (e: IOException) {
            logger.error(e, "Could not add payload $payload to queue: $payloadQueue")
            return
        }

        logger.debug("Enqueue $original payload. ${payloadQueue.size()} elements in the queue. ${Utils.DEFAULT_FLUSH_QUEUE_SIZE}")

        if (payloadQueue.size() >= Utils.DEFAULT_FLUSH_QUEUE_SIZE) {
            submitFlush()
        }

    }

    private fun shouldFlush(): Boolean {
        return payloadQueue.size() > 0 && Utils.isConnected(context)
    }

    private fun submitFlush() {
        if (shouldFlush().not()) {
            return
        }

        performFlush()
    }

    private fun performFlush() {
        logger.verbose("Uploading payloads in queue to Aiactiv.")

        var payloadUploaded = 0
        var connection: HttpsURLConnection? = null

        try {
            val uri = UriUtils.buildUri(Client.TAGS_API_BASE_URI)

            connection = client.openHttpsConnection(uri)
            connection.apply {
                setRequestProperty("Authorization", "$writeKey@android")
                setRequestProperty("Content-Encoding", "gzip")
                setChunkedStreamingMode(0)
                doOutput = true
                connect()
            }

            val outputStream = GZIPOutputStream(connection.outputStream)
            val writer = BatchPayloadWriter(writeKey, outputStream)
                .beginObject()
                .beginBatchArray()

            val payloadWriter = PayloadWriter(writer)
            payloadQueue.forEach(payloadWriter)

            writer.endBatchArray().endObject().close()

            payloadUploaded = payloadWriter.payloadCount

            val response = client.getServiceResponse<String>(connection)
            when {
                response.isSuccess -> {
                    payloadQueue.remove(payloadUploaded)
                    logger.debug("Uploaded $payloadUploaded payloads. ${payloadQueue.size()} remain in the queue.")
                    if (payloadQueue.size() > 0) {
                        performFlush()
                    }
                }
                response.isNetworkError -> logger.debug("Error while uploading payloads.")
                response.isServerError -> logger.debug("Server is temporarily unavailable.")
                response.isInternalError -> logger.debug(response.errorData.getMessage())
            }
        } catch (e: IOException) {
            logger.debug("Error while uploading payloads. $e")
        } finally {
            connection?.disconnect()
        }
    }

    companion object {

        const val AIACTIV_KEY = "aiactiv.io"
        const val AIACTIV_THREAD_NAME = Utils.THREAD_PREFIX + "AiactivDispatcher"

        /**
         * Drop old payloads if queue contains more than 1000 items. Since each item can be at most
         * 32KB, this bounds the queue size to ~32MB (ignoring headers), which also leaves room for
         * QueueFile's 2GB limit.
         */
        const val MAX_QUEUE_SIZE = 1000

        /** Our servers only accept payloads < 32KB.  */
        const val MAX_PAYLOAD_SIZE = 32000 // 32KB.

        /**
         * Our servers only accept batches < 500KB. This limit is 475KB to account for extra data that
         * is not present in payloads themselves, but is added later, such as `sentAt`, `integrations` and other json tokens.
         */
        const val MAX_BATCH_SIZE = 475000 // 475KB.

        val FACTORY = object : Factory {
            override fun create(
                settings: ValueMap?,
                analytics: Analytics
            ): Integration<*> {
                return create(
                    analytics.context,
                    analytics.writeKey,
                    analytics.tag,
                    analytics.logger,
                    analytics.bundledIntegrations,
                    analytics.cartographer,
                )
            }

            override fun key(): String {
                return AIACTIV_KEY
            }
        }

        private fun create(
            context: Context,
            writeKey: String,
            tag: String,
            logger: Logger,
            bundleIntegrations: Map<String, Boolean>,
            cartographer: Cartographer,
        ): AiactivIntegration {
            val payloadQueue: PayloadQueue = try {
                val folder = context.getDir("aiactiv-disk-queue", Context.MODE_PRIVATE)
                val queueFile = createQueueFile(folder, tag)
                PayloadQueue.Companion.PersistenceQueue(queueFile)
            } catch (e: IOException) {
                logger.error(e, "Could not create disk queue. Falling back to memory queue.")
                PayloadQueue.Companion.MemoryQueue()
            }

            return AiactivIntegration(
                context,
                writeKey,
                bundleIntegrations,
                payloadQueue,
                logger,
                cartographer
            )
        }

        private fun createQueueFile(folder: File, name: String): QueueFile {
            Utils.createDirectory(folder)
            val file = File(folder, name)
            return try {
                QueueFile(file)
            } catch (e: IOException) {
                if (file.delete()) {
                    QueueFile(file)
                } else {
                    throw IOException("Could not create queue file $name in $folder.")
                }
            }
        }

        class AiactivDispatcherHandler(looper: Looper, private val integration: AiactivIntegration): Handler(looper) {
            companion object {
                const val REQUEST_FLUSH = 1
                const val REQUEST_ENQUEUE = 2
            }

            override fun handleMessage(msg: Message) {
                when (msg.what) {
                    REQUEST_FLUSH -> integration.submitFlush()
                    REQUEST_ENQUEUE -> integration.performEnqueue(msg.obj as BasePayload)
                    else -> throw AssertionError("Unknown dispatcher message ${msg.what}")
                }
            }
        }


        class PayloadWriter(private val writer: BatchPayloadWriter): PayloadQueue.ElementVisitor {

            var size: Int = 0
            var payloadCount: Int = 0

            override fun read(inputStream: InputStream, length: Int): Boolean {
                val newSize = size + length
                if (newSize > MAX_BATCH_SIZE) {
                    return false
                }
                size = newSize
                val data = ByteArray(length)
                inputStream.read(data, 0, length)
                val str = String(data).trim()
                System.out.println(str)
                writer.emitPayloadObject(String(data).trim())
                payloadCount++
                return true
            }

        }
    }


}