package io.aiactiv.sdk.adnetwork.models.bid

import com.google.gson.annotations.SerializedName

/**
 * This object encapsulates various methods for specifying a geographic location.
 * When subordinate to a Device object, it indicates the location of the device
 * which can also be interpreted as the user’s current location.
 * When subordinate to a User object, it indicates the location of the user’s home base
 * (i.e., not necessarily their current location). The lat/lon attributes should only be passed if
 * they conform to the accuracy depicted in the type attribute.
 * For example, the centroid of a geographic region such as postal code should not be passed.
 */
class Geo {
    /**
     * Latitude from -90.0 to +90.0, where negative is south.
     */
    @field:SerializedName("lat")
    var latitude: Float = 0f

    /**
     * Longitude from -180.0 to +180.0, where negative is west.
     */
    @field:SerializedName("lon")
    var longitude: Float= 0f

    /**
     * Source of location data; recommended when passing lat/lon.
     */
    @field:SerializedName("type")
    var type: GeoType? = null

    /**
     * Estimated location accuracy in meters; recommended when lat/lon are specified and derived
     * from a device’s location services (i.e., type = 1).
     */
    @field:SerializedName("accuracy")
    var accuracy: Int? = null

    /**
     * Number of seconds since this geolocation fix was established. Note that devices may cache location data across multiple fetches.
     * Ideally, this value should be from the time the actual fix was taken.
     */
    @field:SerializedName("lastfix")
    var lastFix: Int? = null

    /**
     * Service or provider used to determine geolocation from IP address if applicable
     */
    @field:SerializedName("ipservice")
    var ipService: IPLocationService? = null

    /**
     * Country code using ISO-3166-1-alpha-3.
     */
    @field:SerializedName("country")
    var country: String? = null

    /**
     * Region code using ISO-3166-2; 2-letter state code if USA.
     */
    @field:SerializedName("region")
    var region: String? = null

    /**
     * Region of a country using FIPS 10-4 notation.
     * While OpenRTB supports this attribute, it has been withdrawn by NIST in 2008.
     */
    @field:SerializedName("regionfips104")
    var regionFIPS104: String? = null

    /**
     * Google metro code; similar to but not exactly Nielsen DMAs.
     */
    @field:SerializedName("metro")
    var metro: String? = null

    /**
     * City using United Nations Code for Trade & Transport Locations.
     */
    @field:SerializedName("city")
    var city: String? = null

    /**
     * Zip or postal code.
     */
    @field:SerializedName("zip")
    var zip: String? = null

    /**
     * Local time as the number +/- of minutes from UTC.
     */
    @field:SerializedName("utcoffset")
    var localTime: Int? = null

    /**
     * Placeholder for exchange-specific extensions to OpenRTB.
     */
    @field:SerializedName("ext")
    var extension: Any? = null

    enum class GeoType(val value: Int) {
        GPS(1),
        IP_ADDRESS(2),
        USER_PROVIDED(3),
    }

    enum class IPLocationService(val value: Int) {
        IP2LOCATION(1),
        NEUSTAR(2),
        MAXMIND(3),
        NET_ACUITY(4),
    }
}