package io.aiactiv.sdk.adnetwork.api

import android.content.Context
import com.google.gson.GsonBuilder
import io.aiactiv.sdk.AdNetwork
import io.aiactiv.sdk.SingletonHolder
import io.aiactiv.sdk.adnetwork.ads.AdRequest
import io.aiactiv.sdk.adnetwork.ads.AdSize
import io.aiactiv.sdk.adnetwork.ads.AdType
import io.aiactiv.sdk.adnetwork.models.batch.BatchType
import io.aiactiv.sdk.adnetwork.models.bid.*
import io.aiactiv.sdk.internal.*
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.asFlow
import kotlinx.coroutines.flow.map
import java.util.*

sealed class ApiResult<out R> {
    data class Success<out T>(val data: T) : ApiResult<T>()
    data class Error(val error: ApiError) : ApiResult<Nothing>()
}

class BidRequestApiClient private constructor(val context: Context) {

    private val serviceHttpClient = Client.getInstance(context)

    private val gson = GsonBuilder().setExclusionStrategies(AnnotationExclusionStrategy()).create()

    private val batchController = BatchController(context, serviceHttpClient, gson)

    suspend fun postBidRequest(
        inventoryId: Int,
        adType: AdType,
        adSize: AdSize?,
        bidRequest: BidRequest
    ): ApiResult<String> {
        val cappingManager = CappingManager.getInstance(context)
        if (cappingManager.isBlackList()) {
            return ApiResult.Error(ApiError("Client is in blacklist"))
        }

        val container = AdNetwork.getInstance(context).container()
            ?: return ApiResult.Error(ApiError("Not found container"))

        val core = container.core

        // Check if Container has Inventory for Ad Unit ID
        val inventory = container.inventories.firstOrNull { it.id == inventoryId }
            ?: return ApiResult.Error(ApiError("Couldn't find any inventory"))

        // Check inventory format
        if (inventory.format != null && inventory.format != adType.value) {
            return ApiResult.Error(ApiError("Inventory not support this format: ${adType.value}"))
        }

        val metadata = inventory.metadata
        if (adType == AdType.BANNER) {
            bidRequest.impression!!.first().apply {
                banner?.apply {
                    width = adSize?.width ?: metadata.width
                    height = adSize?.height ?: metadata.height
                }
            }
        }
        if (adType == AdType.VIDEO) {
            bidRequest.impression!!.first().apply {
                video?.apply {
                    width = adSize?.width ?: metadata.width
                    height = adSize?.height ?: metadata.height
                    mines = metadata.mines ?: listOf("video/mp4")
                    linearity = metadata.linearity
                    skip = metadata.skip
                    skipMin = metadata.skipMin
                    skipAfter = metadata.skipAfter
                    startDelay = metadata.startDelay
                    minBitrate = metadata.minBitrate
                    maxBitrate = metadata.maxBitrate
                    minDuration = metadata.minDuration
                    maxDuration = metadata.maxDuration
                    protocols = metadata.protocols
                    loop = metadata.loop
                }
            }
        }

        // Get Ad Default
        // val adDefault = inventory.defaultAds

        // Check if Inventory has DSP
        val inventoryDsps = inventory.platforms

        // Map DSP information
        val dsps = inventoryDsps?.map { dsp ->
            dsp.url = container.platforms?.find { it.id == dsp.id }?.url
            return@map dsp
        }

        // Create Endpoints for request
        val endpoints = arrayOf(Endpoint("core", core.url, "core"))
        dsps?.forEach { dsp ->
            endpoints.plus(Endpoint(dsp.id, dsp.url!!, "dsp", dsp.directDeal > 0))
        }

        // Perform requests to all endpoints
        val flow = endpoints.asFlow().map { endpoint ->
            val uri = UriUtils.buildUri(endpoint.url)
            if (endpoint.type == "core") {
                // If request to core, append extension info to Impression
                appendImpressionExtension(bidRequest, container.id, inventoryId)
            }
            val body = gson.toJson(bidRequest)
            val response = serviceHttpClient.postWithJson<BidResponse>(uri, emptyMap(), body)

            batchController.addBatch(BatchType.BID_REQUEST, container.configurationID!!, bidRequest)

            //  Return response with endpoint id for collect later
            Pair(endpoint.id, response)
        }

        // Collect success response in 2500 milliseconds
        val successResponses: MutableList<Pair<String, BidResponse>> = mutableListOf()
        withTimeoutOrNull(2500) {
            flow.collect { (id, response) ->
                val bidResponse = response.responseData
                if (response.isSuccess && bidResponse != null) {
                    val pair = Pair(id, bidResponse)
                    successResponses.add(pair)
                    batchController.addBatch(
                        BatchType.BID_RESPONSE,
                        container.configurationID!!,
                        bidResponse
                    )
                }
            }
        }

        val bids = mutableListOf<Bid>()

        // Collect bids from direct deal DSP
        val endpoint = endpoints.firstOrNull { it.directDeal }
        if (endpoint != null) {
            val responseData = successResponses.first { it.first == endpoint.id }.second
            responseData.seatBids?.let { seatBids ->
                for (seatBid in seatBids) {
                    bids.addAll(seatBid.bids)
                }
            }
        } else {
            // Or all bids if there's no direct deal DSP
            for (responseData in successResponses) {
                responseData.second.seatBids?.let { seatBids ->
                    for (seatBid in seatBids) {
                        bids.addAll(seatBid.bids)
                    }
                }
            }
        }

        // Filter by floor price
        val floorPrice = inventory.floorPrice
        val filteredBids = bids.filter { it.price >= floorPrice }.toMutableList()

        // Sort bis by price
        filteredBids.sortByDescending { it.price }

        val chosenBid = bids.firstOrNull()
        if (chosenBid != null) {
            batchController.addBatch(
                BatchType.BID_WINNER,
                container.configurationID!!,
                bidRequest.id
            )
            return ApiResult.Success(chosenBid.adMarkup!!)
        }

        val passBack = inventory.metadata.passBack
        if (!passBack.isNullOrEmpty()) {
            return ApiResult.Success(passBack)
        }


//        if (adDefault !== null && adDefault.seatBids != null) {
//            // Return default ad if no bids return and default
//            val bidsDefault = mutableListOf<Bid>()
//            for (seatBid in adDefault.seatBids) {
//                for (bid in seatBid.bids) {
//                    bidsDefault.add(bid)
//                }
//            }
//            bidsDefault.sortByDescending { it.price }
//            bidsDefault.firstOrNull()?.let {
//                return ApiResult.Success(it.adMarkup!!)
//            }
//        }

        batchController.addBatch(BatchType.BID_ERROR, container.configurationID!!, bidRequest.id)
        return ApiResult.Error(ApiError("No available ads"))
    }

    private fun makeBidRequest(adRequest: AdRequest): BidRequest {
        val bidRequestId = UUID.randomUUID().toString()
        val bidRequest = BidRequest(bidRequestId)
        bidRequest.apply {
            device = Device.getInstance(context)
            app = ApplicationInfo.getInstance(context)
            impression = arrayOf(Impression(id = UUID.randomUUID().toString()))
            user = User()
            extension = adRequest.context
        }

        return bidRequest
    }

    private fun appendImpressionExtension(
        bidRequest: BidRequest,
        containerId: Int,
        inventoryId: Int
    ) {
        bidRequest.impression?.first()?.apply {
            extension = Impression.Extension().apply {
                preBid = PreBid()
                aicactusAds = Impression.Extension.AiCactusAd().apply {
                    this.containerId = containerId
                    this.inventoryId = inventoryId
                    this.headerBids = emptyList()
                }
            }
        }
    }

    fun bannerAdRequest(adRequest: AdRequest): BidRequest {
        val bidRequest = makeBidRequest(adRequest)
        bidRequest.impression!!.first().apply {
            banner = Banner()
        }

        return bidRequest
    }

    fun videoAdRequest(adRequest: AdRequest): BidRequest {
        val bidRequest = makeBidRequest(adRequest)
        bidRequest.impression!!.first().apply {
            video = Video()
        }

        return bidRequest
    }

    fun nativeAdRequest(adAssets: Array<Asset.Request>, adRequest: AdRequest): BidRequest {
        val nativeAdRequest = NativeAdRequest(adAssets)
        val request = gson.toJson(nativeAdRequest)
        val bidRequest = makeBidRequest(adRequest)
        bidRequest.impression!!.first().apply {
            native = Native(request)
        }

        return bidRequest
    }

    fun passBackRequest(url: String): ApiResult<String> {
        val uri = UriUtils.buildUri(url)
        val response = serviceHttpClient.get<String>(uri, emptyMap(), emptyMap())
        response.responseData?.let {
            return ApiResult.Success(it)
        } ?: run {
            return ApiResult.Error(ApiError("Couldn't load passBack"))
        }
    }

    companion object : SingletonHolder<BidRequestApiClient, Context>(::BidRequestApiClient) {
        private val TAG = BidRequestApiClient::class.simpleName
    }
}
