package io.aiactiv.sdk.adnetwork.ads.ima

import android.content.Context
import android.net.Uri
import android.util.AttributeSet
import com.google.ads.interactivemedia.v3.api.*
import com.google.android.exoplayer2.C
import com.google.android.exoplayer2.ExoPlayerFactory
import com.google.android.exoplayer2.SimpleExoPlayer
import com.google.android.exoplayer2.ext.ima.ImaAdsLoader
import com.google.android.exoplayer2.source.MediaSource
import com.google.android.exoplayer2.source.ProgressiveMediaSource
import com.google.android.exoplayer2.source.ads.AdsMediaSource
import com.google.android.exoplayer2.source.dash.DashMediaSource
import com.google.android.exoplayer2.source.hls.HlsMediaSource
import com.google.android.exoplayer2.source.smoothstreaming.SsMediaSource
import com.google.android.exoplayer2.upstream.DataSource
import com.google.android.exoplayer2.upstream.DefaultDataSourceFactory
import com.google.android.exoplayer2.util.Util
import io.aiactiv.sdk.adnetwork.ads.AdRequest
import io.aiactiv.sdk.adnetwork.ads.AdSize
import io.aiactiv.sdk.adnetwork.ads.VideoAdListener
import io.aiactiv.sdk.adnetwork.ads.VideoAdLoader


open class PlayerView(context: Context, attrs: AttributeSet?, defStyleAttr: Int):
    com.google.android.exoplayer2.ui.PlayerView(context, attrs, defStyleAttr),
    AdsLoader.AdsLoadedListener, AdErrorEvent.AdErrorListener, AdEvent.AdEventListener,
        AdsMediaSource.MediaSourceFactory
{

    constructor(context: Context, attrs: AttributeSet?) : this(context, attrs, 0)
    constructor(context: Context): this(context, null, 0)

    private lateinit var dataSourceFactory: DataSource.Factory

    private lateinit var imaAdsLoader: ImaAdsLoader
    private var imaAdsManager: AdsManager? = null

    var contentUri: String? = null
    var autoPlay: Boolean = false

    var listener: ImaAdsLoadListener? = null

    private var contentMediaSource: MediaSource? = null

    fun initializePlayer() {
        this.player = ExoPlayerFactory.newSimpleInstance(context)

        val contentUri = Uri.parse(contentUri)
        dataSourceFactory = DefaultDataSourceFactory(context, Util.getUserAgent(context, "AiactivAdNetwork"))
        contentMediaSource = buildMediaSource(contentUri)

        val simpleExoPlayer = player as SimpleExoPlayer
        simpleExoPlayer.apply {
            seekTo(0)
            playWhenReady = autoPlay
            prepare(contentMediaSource)
        }
    }

    private fun appendMediaSourceWithAds(vastTagUrl: String) {
        val adTagUri = Uri.parse(vastTagUrl)
        imaAdsLoader = ImaAdsLoader(context, adTagUri)
        imaAdsLoader.setPlayer(this.player)
        imaAdsLoader.adsLoader?.apply {
            addAdsLoadedListener(this@PlayerView)
            addAdErrorListener(this@PlayerView)
        }

        val mediaSourceWithAds = AdsMediaSource(contentMediaSource, this, imaAdsLoader, this)
        val simpleExoPlayer = player as SimpleExoPlayer
        simpleExoPlayer.apply {
            seekTo(0)
            playWhenReady = autoPlay
            prepare(mediaSourceWithAds)
        }
    }

    private fun buildMediaSource(uri: Uri?): MediaSource {
        return when (val type = Util.inferContentType(uri)) {
            C.TYPE_DASH -> DashMediaSource.Factory(dataSourceFactory).createMediaSource(uri)
            C.TYPE_SS -> SsMediaSource.Factory(dataSourceFactory).createMediaSource(uri)
            C.TYPE_HLS -> HlsMediaSource.Factory(dataSourceFactory).createMediaSource(uri)
            C.TYPE_OTHER -> ProgressiveMediaSource.Factory(dataSourceFactory).createMediaSource(uri)
            else -> throw IllegalStateException("Unsupported type: $type")
        }
    }

    fun requestAd(adUnitID: Int, adRequest: AdRequest, adSize: AdSize?) {
        val videoAdLoader = VideoAdLoader.Builder(context, adUnitID, adSize).build()

        videoAdLoader.loadAd(adRequest)

        videoAdLoader.listener = object : VideoAdListener {
            override fun onVideoAdLoaded(adUnitID: Int, vastTagUrl: String) {
                appendMediaSourceWithAds(vastTagUrl)
                listener?.onImaAdsLoaded(adUnitID, vastTagUrl)
            }

            override fun onVideoAdFailedToLoad(adUnitID: Int, error: String) {
                listener?.onImaAdsFailedToLoad(adUnitID, error)
            }
        }
    }

    fun releasePlayer() {
        if (this::imaAdsLoader.isInitialized) {
            imaAdsLoader.setPlayer(null)
        }
        player?.release()
        player = null
    }

    override fun onAdsManagerLoaded(adsManagerLoadedEvent: AdsManagerLoadedEvent?) {
        imaAdsManager = adsManagerLoadedEvent?.adsManager
        imaAdsManager?.apply {
            addAdErrorListener(this@PlayerView)
            addAdEventListener(this@PlayerView)
            init()
        }
    }

    override fun onAdError(adErrorEvent: AdErrorEvent?) {
        listener?.onAdErrorEvent(adErrorEvent)
    }

    override fun onAdEvent(adEvent: AdEvent?) {
        listener?.onAdEvent(adEvent)
    }

    /**
     * Creates a new [MediaSource] for loading the ad media with the specified `uri`.
     *
     * @param uri The URI of the media or manifest to play.
     * @return The new media source.
     */
    override fun createMediaSource(uri: Uri?): MediaSource {
        return buildMediaSource(uri)
    }

    /**
     * Returns the content types supported by media sources created by this factory. Each element
     * should be one of [C.TYPE_DASH], [C.TYPE_SS], [C.TYPE_HLS] or [ ][C.TYPE_OTHER].
     *
     * @return The content types supported by media sources created by this factory.
     */
    override fun getSupportedTypes(): IntArray {
        return intArrayOf(C.TYPE_DASH, C.TYPE_HLS, C.TYPE_OTHER)
    }
}
