package io.aiactiv.sdk.adnetwork.ads

import android.content.Context
import android.graphics.Canvas
import android.graphics.Color
import android.graphics.Paint
import android.graphics.Rect
import android.util.AttributeSet
import android.widget.FrameLayout
import androidx.core.content.ContextCompat
import androidx.core.content.withStyledAttributes
import androidx.core.view.children
import io.aiactiv.sdk.R

/**
 * The View to display banner ads. The ad size and ad unit ID must be set prior to calling
 */
abstract class BaseAdView @JvmOverloads constructor(
    context: Context,
    attributeSet: AttributeSet? = null,
    defStyleAttr: Int = 0,
) : FrameLayout(context, attributeSet, defStyleAttr) {

    var unitId: Int? = null
        set(value) {
            field = value
            applyDefaults()
            invalidate()
        }

    var size: AdSize? = null
        set(value) {
            field = value
            applyDefaults()
            invalidate()
        }

    var type: AdType = AdType.BANNER

    private val borderPaint = Paint().apply {
        color = Color.BLACK
        strokeWidth = 1.0f
        style = Paint.Style.STROKE
    }

    private val textPaint = Paint(Paint.ANTI_ALIAS_FLAG).apply {
        color = Color.BLACK
        textSize = 48.0f
    }

    private lateinit var title: String
    private val titleRect = Rect()

    init {
        context.withStyledAttributes(attributeSet, R.styleable.BaseAdView) {
            unitId = getInteger(R.styleable.BaseAdView_acAdUnitId, -1)

            val adSize = getInt(R.styleable.BaseAdView_acAdSize, -1)
            if (adSize > -1) {
                size = AdSize.values()[adSize]
            }
        }
        applyDefaults()
    }

    private fun applyDefaults() {
        title = "Ads by Aicactus"
        if (unitId == null || unitId!! < 0) {
            title = "Missing Unit Ad ID"
        }

        if (size == null) {
            title = "Missing Ad Size"
        }

        if (isInEditMode && (unitId == null || size == null)) {
            textPaint.color = Color.WHITE
            setBackgroundColor(ContextCompat.getColor(context, android.R.color.holo_red_dark))
        } else {
            textPaint.color = Color.BLACK
            setBackgroundColor(ContextCompat.getColor(context, android.R.color.darker_gray))
        }
    }

    override fun onDraw(canvas: Canvas?) {
        canvas?.let {
            textPaint.getTextBounds(title, 0, title.length, titleRect)
            canvas.drawText(
                title,
                (width / 2).toFloat() - titleRect.exactCenterX(),
                (height / 2).toFloat() - titleRect.exactCenterY(),
                textPaint,
            )
            canvas.drawRect(
                0f,
                0f,
                width.toFloat() - 1f,
                height.toFloat() - 1f,
                borderPaint
            )
        }
    }

    override fun onMeasure(widthMeasureSpec: Int, heightMeasureSpec: Int) {
        super.onMeasure(widthMeasureSpec, heightMeasureSpec)

        val density = resources.displayMetrics.density

        var widthSize = (MINIMUM_WIDTH * density).toInt()
        var heightSize = (MINIMUM_HEIGHT * density).toInt()

        size?.let { size ->
            val widthPixels = resources.displayMetrics.widthPixels
            widthSize = (size.width * density).toInt().coerceAtMost(widthPixels)
            heightSize = (size.height * density).toInt()
        }

        setMeasuredDimension(widthSize, heightSize)

        for (child in children) {
            child.measure(
                MeasureSpec.makeMeasureSpec(measuredWidth, MeasureSpec.EXACTLY),
                MeasureSpec.makeMeasureSpec(measuredHeight, MeasureSpec.EXACTLY),
            )
        }
    }

    abstract fun loadAd(adRequest: AdRequest)

    companion object {
        const val MINIMUM_WIDTH = 320
        const val MINIMUM_HEIGHT = 50
    }

}