package io.aiactiv.sdk.adnetwork.ads

import android.content.Context
import io.aiactiv.sdk.SingletonHolder
import io.aiactiv.sdk.adnetwork.api.BidRequest
import io.aiactiv.sdk.adnetwork.api.BidRequestApiClient
import io.aiactiv.sdk.adnetwork.models.bid.Asset
import kotlinx.coroutines.*

class AdViewController(val context: Context) {

    private val bidRequestApiClient = BidRequestApiClient.getInstance(context)

    private val requestJob = Job()
    private val scope = CoroutineScope(Dispatchers.IO + requestJob)

    fun request(
        adUnitID: Int,
        adSize: AdSize?,
        adType: AdType,
        adRequest: AdRequest,
        callback: (io.aiactiv.sdk.adnetwork.api.ApiResult<String>) -> Unit
    ) {
        val request = if (adType === AdType.BANNER)
            bidRequestApiClient.bannerAdRequest(adRequest)
        else bidRequestApiClient.videoAdRequest(adRequest)
        postRequestJob(adUnitID, request, adType, adSize, callback)
    }

    fun requestNativeAd(
        adUnitID: Int,
        adAssets: Array<Asset.Request>,
        adRequest: AdRequest,
        adSize: AdSize?,
        callback: (io.aiactiv.sdk.adnetwork.api.ApiResult<String>) -> Unit,
    ) {
        val request = bidRequestApiClient.nativeAdRequest(adAssets, adRequest)
        postRequestJob(adUnitID, request, AdType.NATIVE, adSize, callback)
    }

    fun requestPassBack(url: String, callback: (io.aiactiv.sdk.adnetwork.api.ApiResult<String>) -> Unit) {
        scope.launch {
            val response = bidRequestApiClient.passBackRequest(url)
            withContext(Dispatchers.Main) {
                callback(response)
            }
        }
    }

    private fun postRequestJob(
        adUnitID: Int,
        request: BidRequest,
        type: AdType,
        adSize: AdSize?,
        callback: (io.aiactiv.sdk.adnetwork.api.ApiResult<String>) -> Unit
    ) {
        scope.launch {
            val response = bidRequestApiClient.postBidRequest(adUnitID, type, adSize, request)
            withContext(Dispatchers.Main) {
                callback(response)
            }
        }
    }

    companion object : SingletonHolder<AdViewController, Context>(::AdViewController) {
        private val TAG = AdViewController::class.simpleName
    }
}
