package io.aiactiv.sdk.adnetwork.ads

import android.annotation.SuppressLint
import android.content.Context
import android.content.Intent
import android.net.Uri
import android.util.AttributeSet
import android.util.Log
import android.view.ViewGroup
import android.webkit.URLUtil
import android.webkit.WebResourceRequest
import android.webkit.WebView
import android.webkit.WebViewClient
import io.aiactiv.sdk.adnetwork.api.ApiResult

class AdView(context: Context, attributeSet: AttributeSet?) : BaseAdView(context, attributeSet) {

    constructor(context: Context): this(context, null)

    private var adViewController = AdViewController.getInstance(context)

    private lateinit var webView: WebView

    var adListener: AdListener? = null

    override fun loadAd(adRequest: AdRequest) {
        if (unitId != null && size != null) {
            adViewController.request(unitId!!, size, type, adRequest) { response ->
                when(response) {
                    is ApiResult.Success<String> -> {
                        val adMarkup = response.data
                        if( URLUtil.isValidUrl(adMarkup)) {
                            adViewController.requestPassBack(adMarkup) { res ->
                                when (res) {
                                    is ApiResult.Success<String> -> {
                                        val passBackData = res.data
                                        attachView(passBackData)
                                        adListener?.onAdLoaded(unitId!!)
                                    }
                                    else -> {
                                        val error = (res as ApiResult.Error).error
                                        val message = "Load Ad failed: ${error.getMessage()}"
                                        Log.e("AdView", message)
                                        adListener?.onAdFailedToLoad(unitId!!, message)
                                    }
                                }
                            }
                        } else {
                            attachView(adMarkup)
                            adListener?.onAdLoaded(unitId!!)
                        }
                    }
                    else -> {
                        val error = (response as ApiResult.Error).error
                        val message = "Load Ad failed: ${error.getMessage()}"
                        Log.e("AdView", message)
                        adListener?.onAdFailedToLoad(unitId!!, message)
                    }
                }
            }
        } else {
            val message = "Load Ad failed: missing Ad unit ID or size"
            Log.e("AdView", message)
            unitId?.let { adListener?.onAdFailedToLoad(it,message) } ?: run {
                adListener?.onAdFailedToLoad(-1, message)
            }
        }
    }

    @SuppressLint("SetJavaScriptEnabled", "ClickableViewAccessibility")
    private fun attachView(adMarkup: String) {
        val html = createHTML(adMarkup)

        try {
            webView = WebView(context)
            webView.apply {
                webViewClient = object : WebViewClient() {
                    override fun shouldOverrideUrlLoading(
                        view: WebView?,
                        request: WebResourceRequest?
                    ): Boolean {
                        request?.let {
                            this@AdView.adListener?.let { listener ->
                                val result = listener.shouldOverrideOnAdClicked(this@AdView, it.url)
                                if (result) {
                                    val intent = Intent(Intent.ACTION_VIEW, it.url)
                                    this@AdView.context.startActivity(intent)
                                }
                            }
                        }
                        return true
                    }

                    @Deprecated("Deprecated in Java")
                    override fun shouldOverrideUrlLoading(view: WebView?, url: String?): Boolean {
                        this@AdView.adListener?.let { listener ->
                            val result = listener.shouldOverrideOnAdClicked(this@AdView, Uri.parse(url))
                            if (result) {
                                val intent = Intent(Intent.ACTION_VIEW, Uri.parse(url))
                                this@AdView.context.startActivity(intent)
                            }
                        }
                        return true
                    }
                }
                settings.apply {
                    javaScriptEnabled = true
                    mediaPlaybackRequiresUserGesture = false
                }
                isVerticalScrollBarEnabled = false
                isHorizontalScrollBarEnabled = false
            }
            webView.layoutParams = ViewGroup.LayoutParams(width, height)
            webView.loadDataWithBaseURL(null, html, "text/html", null, null)
            addView(webView)
        } catch (e: Exception) {
            e.message?.let { message ->
                if (message.contains("MissingWebViewPackageException")
                    || message.contains("No WebView installed")
                    || message.contains("WebView")) {
                    Log.d("Device", "No WebView installed")
                } else {
                    Log.d("Device", message)
                }
            }
        }
    }

    private fun createHTML(adMarkup: String): String {
        return "<!DOCTYPE html>" +
                "<html>" +
                "<meta name=\"viewport\" content=\"width=$width, shrink-to-fit=YES\">" +
                "<body style=\"margin: 0; padding: 0\">$adMarkup</body>" +
                "</html>"
    }
}