package io.aiactiv.sdk.adnetwork.ads

import android.content.Context
import android.util.Log
import com.google.gson.Gson
import io.aiactiv.sdk.adnetwork.models.bid.*

class AdLoader(
    context: Context,
    private val adUnitID: Int,
    private val callback: ((nativeAd: NativeAdResponse) -> Unit)? = null,
) {

    private var adViewController = AdViewController.getInstance(context)

    fun loadAd(adRequest: AdRequest) {
        val mainImage = Asset.Request().apply {
            image = AssetImage.Request().apply {
                type = AssetImage.Type.MAIN
                minWidth = 300
                minHeight = 250
            }
            required = 1
        }

        val iconImage = Asset.Request().apply {
            image = AssetImage.Request().apply {
                type = AssetImage.Type.ICON
                minWidth = 40
                minHeight = 40
            }
            required = 1
        }

        val title = Asset.Request().apply {
            title = AssetTitle.Request(140)
            required = 1
        }

        val body = Asset.Request().apply {
            data = AssetData.Request().apply {
                type = AssetData.Type.DESCRIPTION
            }
        }

        val cta = Asset.Request().apply {
            data = AssetData.Request().apply {
                type = AssetData.Type.CTA_DESCRIPTION
            }
        }

        val sponsored = Asset.Request().apply {
            data = AssetData.Request().apply {
                type = AssetData.Type.SPONSORED
            }
        }

        val assets = arrayOf(mainImage, iconImage, title, body, cta, sponsored)
        adViewController.requestNativeAd(adUnitID, assets, adRequest, null) { response ->
            when (response) {
                is io.aiactiv.sdk.adnetwork.api.ApiResult.Success<String> -> {
                    val adMarkup = response.data
                    try {
                        val nativeAd = Gson().fromJson(adMarkup, NativeAdResponse::class.java)
                        callback?.let { callback ->
                            callback(nativeAd)
                        }
                    } catch (e: Exception) {
                        Log.e(
                            "AdLoader",
                            "Load Ad failed: Cannot load Ad Markup due to ${e.localizedMessage}"
                        )
                    }
                }
                else -> {
                    val error = (response as io.aiactiv.sdk.adnetwork.api.ApiResult.Error).error
                    Log.e("AdLoader", "Load Ad failed: ${error.getMessage()}")
                }
            }
        }
    }

    companion object {
        inline fun build(context: Context, adUnitID: Int, block: Builder.() -> Unit) =
            Builder(context, adUnitID).apply(block).build()
    }

    class Builder(
        private val context: Context,
        private val adUnitID: Int,
    ) {
        private var callback: ((nativeAd: NativeAdResponse) -> Unit)? = null

        fun build() = AdLoader(context, adUnitID, callback)

        fun forNativeAd(callback: (nativeAd: NativeAdResponse) -> Unit): Builder {
            this.callback = callback
            return this
        }
    }
}
