package io.aiactiv.sdk

import io.aiactiv.sdk.internal.PayloadQueue
import java.io.*
import kotlin.experimental.and
import kotlin.math.min

internal class QueueFile(file: File): Closeable {

    /**
     * The underlying file. Uses a ring buffer to store entries. Designed so that a modification
     * isn't committed or visible until we write the header. The header is much smaller than a
     * segment. So long as the underlying file system supports atomic segment writes, changes to the
     * queue are atomic. Storing the file length ensures we can recover from a failed expansion
     * (i.e. if setting the file length succeeds but the process dies before the data can be
     * copied).
     *
     * <p>
     *
     * <pre>
     *   Format:
     *     Header              (16 bytes)
     *     Element Ring Buffer (File Length - 16 bytes)
     * <p/>
     *   Header:
     *     File Length            (4 bytes)
     *     Element Count          (4 bytes)
     *     First Element Position (4 bytes, =0 if null)
     *     Last Element Position  (4 bytes, =0 if null)
     * <p/>
     *   Element:
     *     Length (4 bytes)
     *     Data   (Length bytes)
     * </pre>
     *
     * Visible for testing.
     */
    private var raf: RandomAccessFile

    /** Cached file length. Always a power of 2.  */
    private var fileLength = 0

    /** Number of elements.  */
    private var elementCount = 0

    /** Pointer to first (or eldest) element.  */
    private lateinit var first: Element

    /** Pointer to last (or newest) element.  */
    private lateinit var last: Element


    init {
        if (!file.exists()) {
            initialize(file)
        }
        raf = RandomAccessFile(file, "rwd")
        readHeader()
    }

    @Synchronized
    override fun close() {
        raf.close()
    }

    @Throws(IOException::class)
    fun readHeader() {
        raf.seek(0)
        raf.readFully(BUFFER)
        fileLength = readInt(BUFFER, 0)
        elementCount = readInt(BUFFER, 4)
        val firstOffset = readInt(BUFFER, 8)
        val lastOffset = readInt(BUFFER, 12)

        if (fileLength > raf.length()) {
            throw IOException("File is truncated. Expected length: $fileLength, Actual length: ${raf.length()}.")
        } else if (fileLength <= 0) {
            throw IOException("File is corrupt. Length stored in header ($fileLength) is invalid.")
        } else if (firstOffset < 0 || fileLength <= wrapPosition(firstOffset)) {
            throw IOException("File is corrupt. first position store in header ($firstOffset) is invalid.")
        } else if (lastOffset < 0 || fileLength <= wrapPosition(lastOffset)) {
            throw IOException("File is corrupt. last position store in header ($lastOffset) is invalid.")
        }

        first = readElement(firstOffset)
        last = readElement(lastOffset)
    }

    @Throws(IOException::class)
    private fun readElement(position: Int): Element {
        if (position == 0) return Element.NULL
        ringRead(position, BUFFER, 0, Element.HEADER_LENGTH)
        val length = readInt(BUFFER, 0)
        System.out.println("readElement length = $length")
        return Element(position, length)
    }

    @Throws(IOException::class)
    private fun ringWrite(position: Int, buffer: ByteArray, offset: Int, count: Int) {
        val pos = wrapPosition(position)
        if (pos + count <= fileLength) {
            raf.seek(pos.toLong())
            raf.write(buffer, offset, count)
        } else {
            val beforeEOF = fileLength - pos
            raf.seek(pos.toLong())
            raf.write(buffer, offset, beforeEOF)
            raf.seek(HEADER_LENGTH.toLong())
            raf.write(buffer, offset + beforeEOF, count - beforeEOF)
        }
    }

    @Throws(IOException::class)
    private fun ringErase(position: Int, length: Int) {
        var len = length
        var pos = position
        while (len > 0) {
            val chunk = min(length, ZEROES.size)
            ringWrite(position, ZEROES, 0, chunk)
            len -= chunk
            pos += chunk
        }
    }

    @Throws(IOException::class)
    private fun ringRead(position: Int, buffer: ByteArray, offset: Int, count: Int) {
        val pos = wrapPosition(position)
        if (pos + count <= fileLength) {
            raf.seek(pos.toLong())
            raf.readFully(buffer, offset, count)
        } else {
            val beforeEof = fileLength - pos
            raf.seek(pos.toLong())
            raf.readFully(buffer, offset, beforeEof)
            raf.seek(HEADER_LENGTH.toLong())
            raf.readFully(buffer, offset + beforeEof, count - beforeEof)
        }
    }

    @Throws(IOException::class)
    fun add(data: ByteArray) {
        add(data, 0, data.size)
    }

    @Synchronized
    @Throws(IOException::class)
    fun add(data: ByteArray, offset: Int, count: Int) {
        if ((offset or count) < 0 || count > data.size - offset) {
            throw IndexOutOfBoundsException()
        }

        expandIfNecessary(count)

        val wasEmpty = isEmpty()
        val position = if (wasEmpty) HEADER_LENGTH else wrapPosition(last.position + Element.HEADER_LENGTH + last.length)
        val newLast = Element(position, count)

        writeInt(BUFFER, 0, count)
        ringWrite(newLast.position, BUFFER, 0, Element.HEADER_LENGTH)
        ringWrite(newLast.position + Element.HEADER_LENGTH, data, offset, count)

        val firstPosition = if (wasEmpty) newLast.position else first.position
        writeHeader(fileLength, elementCount + 1, firstPosition, newLast.position)
        last = newLast
        elementCount++
        if (wasEmpty) first = last
    }

    private fun useBytes(): Int {
        if (elementCount == 0) return HEADER_LENGTH
        return if (last.position >= first.position) {
            (last.position - first.position) + Element.HEADER_LENGTH + last.length + HEADER_LENGTH
        } else {
            last.position + Element.HEADER_LENGTH + last.length + fileLength - first.position
        }
    }

    private fun remainingBytes() = fileLength - useBytes()

    @Synchronized
    @Throws(IOException::class)
    private fun expandIfNecessary(dataLength: Int) {
        val elementLength = Element.HEADER_LENGTH + dataLength
        var remainingBytes = remainingBytes()
        if (remainingBytes >= elementLength) return

        var previousLength = fileLength
        var newLength: Int
        do {
            remainingBytes += previousLength
            newLength = previousLength shl 1
            if (newLength < previousLength) {
                throw EOFException("Cannot grow file beyond $previousLength bytes.")
            }
            previousLength = newLength
        } while (remainingBytes < elementLength)

        setLength(newLength)

        val endOfLastElement = wrapPosition(last.position + Element.HEADER_LENGTH + last.length)
        if (endOfLastElement <= first.position) {
            val channel = raf.channel
            channel.position(fileLength.toLong())
            val count = endOfLastElement - HEADER_LENGTH
            if (channel.transferTo(HEADER_LENGTH.toLong(), count.toLong(), channel) != count.toLong()) {
                throw AssertionError("Copied insufficient number of bytes!")
            }
            ringErase(HEADER_LENGTH, count)
        }

        if (last.position < first.position) {
            val newLastPosition = fileLength + last.position - HEADER_LENGTH
            writeHeader(newLength, elementCount, first.position, newLastPosition)
            last = Element(newLastPosition, last.length)
        } else {
            writeHeader(newLength, elementCount, first.position, last.position)
        }

        fileLength = newLength
    }


    @Throws(IOException::class)
    private fun writeHeader(fileLength: Int, elementCount: Int, firstPosition: Int, lastPosition: Int) {
        writeInt(BUFFER, 0, fileLength)
        writeInt(BUFFER, 4, elementCount)
        writeInt(BUFFER, 8, firstPosition)
        writeInt(BUFFER, 12, lastPosition)
        raf.seek(0)
        raf.write(BUFFER)
    }

    private fun wrapPosition(position: Int): Int {
        return if (position < fileLength) position else (HEADER_LENGTH + position - fileLength)
    }

    @Throws(IOException::class)
    private fun setLength(newLength: Int) {
        raf.setLength(newLength.toLong())
        raf.channel.force(true)
    }

    companion object {

        /** Initial file size in bytes.  */
        private const val INITIAL_LENGTH = 4096 // one file system block

        /** A block of nothing to write over old data.  */
        private val ZEROES = ByteArray(INITIAL_LENGTH)

        /** Length of header in bytes.  */
        const val HEADER_LENGTH = 16

        /** In-memory buffer. Big enough to hold the header.  */
        private val BUFFER = ByteArray(16)

        @Throws(IOException::class)
        fun initialize(file: File) {
            val tempFile = File(file.path + ".tmp")
            val raf = RandomAccessFile(tempFile, "rwd")
            raf.use {
                it.setLength(INITIAL_LENGTH.toLong())
                it.seek(0)
                val headerBuffer = ByteArray(16)
                writeInt(headerBuffer, 0, INITIAL_LENGTH)
                it.write(headerBuffer)
            }

            if (!tempFile.renameTo(file)) {
                throw IOException("Rename failed!")
            }
        }

        private fun writeInt(buffer: ByteArray, offset: Int, value: Int) {
            buffer[offset] = (value shr 24).toByte()
            buffer[offset + 1] = (value shr 16).toByte()
            buffer[offset + 2] = (value shr 8).toByte()
            buffer[offset + 3] = value.toByte()
        }

        private fun readInt(buffer: ByteArray, offset: Int): Int {
            return (((buffer[offset] and 0xff.toByte()).toInt() shl 24)
                    + ((buffer[offset + 1] and 0xff.toByte()).toInt() shl 16)
                    + ((buffer[offset + 2] and 0xff.toByte()).toInt() shl 8)
                    + (buffer[offset + 3] and 0xff.toByte()))
        }

        class Element(val position: Int, val length: Int) {

            override fun toString(): String {
                return "${Element::javaClass.name}[position = $position, length = $length]"
            }

            companion object {
                const val HEADER_LENGTH = 4
                val NULL = Element(0, 0)
            }
        }

    }

    inner class ElementInputStream(element: Element): InputStream() {

        private var position = wrapPosition(element.position + Element.HEADER_LENGTH)
        private var remaining = element.length

        override fun read(buffer: ByteArray, offset: Int, length: Int): Int {
            if ((offset or length) < 0 || length > (buffer.size - offset)) {
                throw ArrayIndexOutOfBoundsException()
            }
            if (remaining == 0) {
                return -1
            }
            var len = length
            if (length > remaining) len = remaining
            ringRead(position, buffer, offset, len)
            position = wrapPosition(position + len)
            remaining -= len
            return len
        }

        override fun read(): Int {
            if (remaining == 0) return -1
            raf.seek(position.toLong())
            val b = raf.read()
            position = wrapPosition(position + 1)
            remaining--
            return b
        }
    }

    @Synchronized
    @Throws(IOException::class)
    fun forEach(reader: PayloadQueue.ElementVisitor): Int {
        var position = first.position
        for (i in 0 until elementCount) {
            val current = readElement(position)
            System.out.println("Element count $i ${current.length}")
            val shouldContinue = reader.read(ElementInputStream(current), current.length)
            if (shouldContinue.not()) {
                return i + 1
            }
            position = wrapPosition(current.position + Element.HEADER_LENGTH + current.length)
        }
        return elementCount
    }

    @Synchronized
    fun size(): Int {
        return elementCount
    }

    @Synchronized
    fun isEmpty(): Boolean = elementCount == 0

    @Synchronized
    @Throws(IOException::class)
    fun remove() {
        remove(1)
    }

    @Synchronized
    @Throws(IOException::class)
    fun remove(n: Int)  {
        when {
            isEmpty() -> throw NoSuchElementException()
            n < 0 -> throw IllegalArgumentException("Cannot remove negative $n number of elements.")
            n == 0 -> return
            n == elementCount -> {
                clear()
                return
            }
            n > elementCount -> throw IllegalArgumentException("Cannot remove more elements ($n) than present in queue ($elementCount).")
        }

        val eraseStartPosition = first.position
        var eraseTotalLength = 0

        var newFirstPosition = first.position
        var newFirstLength = first.length

        for (i in 0 until n) {
            eraseTotalLength += Element.HEADER_LENGTH + newFirstLength
            newFirstPosition = wrapPosition(newFirstPosition + Element.HEADER_LENGTH + newFirstLength)
            ringRead(newFirstPosition, BUFFER, 0, Element.HEADER_LENGTH)
            newFirstLength = readInt(BUFFER, 0)
        }

        writeHeader(fileLength, elementCount - n, newFirstPosition, last.position)
        elementCount -= n
        first = Element(newFirstPosition, newFirstLength)
        ringErase(eraseStartPosition, eraseTotalLength)
    }

    @Synchronized
    @Throws(IOException::class)
    fun clear() {
        writeHeader(INITIAL_LENGTH, 0, 0, 0)
        raf.seek(HEADER_LENGTH.toLong())
        raf.write(ZEROES, 0, INITIAL_LENGTH - HEADER_LENGTH)
        elementCount = 0
        first = Element.NULL
        last = Element.NULL
        if (fileLength > INITIAL_LENGTH) setLength(INITIAL_LENGTH)
        fileLength = INITIAL_LENGTH
    }
}