package io.aiactiv.sdk

import android.app.Activity
import android.os.Bundle
import io.aiactiv.sdk.analytics.AiactivIntegration
import io.aiactiv.sdk.analytics.integrations.*
import io.aiactiv.sdk.analytics.integrations.Integration
import io.aiactiv.sdk.internal.ValueMap

internal abstract class IntegrationOperation {

    abstract fun run(key: String, integration: Integration<*>, projectSettings: ProjectSettings)

    companion object {

        fun aiactivEvent(payload: BasePayload, destinationMiddleware: Map<String, List<Middleware>>): IntegrationOperation {
            return object : IntegrationOperation() {
                override fun run(
                    key: String,
                    integration: Integration<*>,
                    projectSettings: ProjectSettings
                ) {
                    val applicableMiddleware = getMiddlewareList(destinationMiddleware, key)
                    runMiddlewareChain(payload, applicableMiddleware, object : Middleware.Callback {
                        override fun invoke(payload: BasePayload) {
                            when (payload.type()) {
                                BasePayload.Type.track -> track(payload as TrackPayload, key, integration, projectSettings)
                                BasePayload.Type.identity -> identify(payload as IdentifyPayload, key, integration)
                                BasePayload.Type.screen -> screen(payload as ScreenPayload, key, integration)
                                else -> throw AssertionError("unknown type ${payload.type()}")
                            }
                        }
                    })
                }

                override fun toString(): String {
                    return payload.toString()
                }
            }
        }

        fun onActivityCreated(activity: Activity, bundle: Bundle?): IntegrationOperation {
            return object : IntegrationOperation() {
                override fun run(
                    key: String,
                    integration: Integration<*>,
                    projectSettings: ProjectSettings
                ) {
                    integration.onActivityCreated(activity, bundle)
                }

                override fun toString(): String {
                    return "Activity Created"
                }
            }
        }

        fun onActivityStarted(activity: Activity): IntegrationOperation {
            return object : IntegrationOperation() {
                override fun run(
                    key: String,
                    integration: Integration<*>,
                    projectSettings: ProjectSettings
                ) {
                    integration.onActivityStarted(activity)
                }

                override fun toString(): String {
                    return "Activity Started"
                }
            }
        }

        fun onActivityResumed(activity: Activity): IntegrationOperation {
            return object : IntegrationOperation() {
                override fun run(
                    key: String,
                    integration: Integration<*>,
                    projectSettings: ProjectSettings
                ) {
                    integration.onActivityResumed(activity)
                }

                override fun toString(): String {
                    return "Activity Resumed"
                }
            }
        }

        fun onActivityPaused(activity: Activity): IntegrationOperation {
            return object : IntegrationOperation() {
                override fun run(
                    key: String,
                    integration: Integration<*>,
                    projectSettings: ProjectSettings
                ) {
                    integration.onActivityPaused(activity)
                }

                override fun toString(): String {
                    return "Activity Paused"
                }
            }
        }

        fun onActivityStopped(activity: Activity): IntegrationOperation {
            return object : IntegrationOperation() {
                override fun run(
                    key: String,
                    integration: Integration<*>,
                    projectSettings: ProjectSettings
                ) {
                    integration.onActivityStopped(activity)
                }

                override fun toString(): String {
                    return "Activity Stopped"
                }
            }
        }

        fun onActivitySaveInstanceState(activity: Activity, bundle: Bundle?): IntegrationOperation {
            return object : IntegrationOperation() {
                override fun run(
                    key: String,
                    integration: Integration<*>,
                    projectSettings: ProjectSettings
                ) {
                    integration.onActivitySaveInstanceState(activity, bundle)
                }

                override fun toString(): String {
                    return "Activity Save Instance"
                }
            }
        }

        fun onActivityDestroyed(activity: Activity): IntegrationOperation {
            return object : IntegrationOperation() {
                override fun run(
                    key: String,
                    integration: Integration<*>,
                    projectSettings: ProjectSettings
                ) {
                    integration.onActivityDestroyed(activity)
                }

                override fun toString(): String {
                    return "Activity Destroyed"
                }
            }
        }

        private fun track(payload: TrackPayload, key: String, integration: Integration<*>, projectSettings: ProjectSettings) {
            val integrationOptions = payload.integrations()
            val trackingPlan = projectSettings.trackingPlan()

            if (trackingPlan.isNullOrEmpty()) {
                if (isIntegrationEnabled(integrationOptions, key)) {
                    integration.track(payload)
                }
                return
            }

            val event = payload.event()
            val eventPlan = trackingPlan.getValueMap(event)
            if (eventPlan.isNullOrEmpty()) {
                if (integrationOptions.isNullOrEmpty().not()) {
                    if (isIntegrationEnabled(integrationOptions, key)) {
                        integration.track(payload)
                    }
                    return
                }

                val defaultPlan = trackingPlan.getValueMap("__default")
                if (defaultPlan.isNullOrEmpty()) {
                    integration.track(payload)
                    return
                }

                val defaultEventsEnabled = defaultPlan.getBoolean("enabled", true)
                if (defaultEventsEnabled || AiactivIntegration.AIACTIV_KEY == key) {
                    integration.track(payload)
                }

                return
            }

            val isEnabled = eventPlan.getBoolean("enabled", true)
            if (isEnabled.not()) {
                if (AiactivIntegration.AIACTIV_KEY == key) {
                    integration.track(payload)
                }
                return
            }

            val integrations = ValueMap()
            val eventIntegrations = eventPlan.getValueMap("integrations")
            if (eventIntegrations != null) {
                integrations.putAll(eventIntegrations)
            }
            if (integrationOptions != null) {
                integrations.putAll(integrationOptions)
            }
            if (isIntegrationEnabled(integrations, key)) {
                integration.track(payload)
            }
        }

        private fun identify(identifyPayload: IdentifyPayload, key: String, integration: Integration<*>) {
            if (isIntegrationEnabled(identifyPayload.integrations(), key)) {
                integration.identify(identifyPayload)
            }
        }

        private fun screen(screenPayload: ScreenPayload, key: String, integration: Integration<*>) {
            if (isIntegrationEnabled(screenPayload.integrations(), key)) {
                integration.screen(screenPayload)
            }
        }

        private fun runMiddlewareChain(payload: BasePayload, middleware: List<Middleware>, callback: Middleware.Callback) {
            val chain = MiddleChainRunner(0, payload, middleware, callback)
            chain.proceed(payload)
        }

        private fun getMiddlewareList(destinationMiddleware: Map<String, List<Middleware>>, key: String): List<Middleware> {
            return destinationMiddleware[key] ?: emptyList()
        }

        private fun isIntegrationEnabled(integrations: ValueMap?, key: String): Boolean {
            if (integrations.isNullOrEmpty()) {
                return true
            }

            if (AiactivIntegration.AIACTIV_KEY == key) {
                return true
            }

            var enabled = true
            if (integrations.containsKey(key)) {
                enabled = integrations.getBoolean(key, true)
            } else if (integrations.containsKey(Options.ALL_INTEGRATIONS_KEY)) {
                enabled = integrations.getBoolean(Options.ALL_INTEGRATIONS_KEY, true)
            }
            return enabled;
        }
    }
}