package io.aiactiv.sdk

import android.content.Context
import android.content.pm.PackageManager
import android.os.Build
import android.os.Bundle

class Configuration(
    val writeKey: String,
    val writeKeyForAdNetwork: String?,
) {
    companion object {
        private const val WRITE_KEY_METADATA = "io.aiactiv.sdk.WRITE_KEY"
        private const val WRITE_KEY_ADNETWORK_METADATA = "io.aiactiv.sdk.WRITE_KEY_ADNETWORK"

        fun build(context: Context, block: Builder.() -> Unit): Configuration {
            return Builder(context).apply(block).build()
        }
    }

    class Builder(private val context: Context) {

        private fun getAppBundle(): Bundle {
            val packageManager = context.packageManager
            val packageName = context.packageName

            @Suppress("DEPRECATION")
            val applicationInfo = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
                packageManager.getApplicationInfo(packageName, PackageManager.ApplicationInfoFlags.of(PackageManager.GET_META_DATA.toLong()))
            } else {
                packageManager.getApplicationInfo(packageName, PackageManager.GET_META_DATA)
            }

            return applicationInfo.metaData
        }

        private fun getWriteKey(): Result<String> {
            return runCatching {
                val appBundle = getAppBundle()
                val writeKey = appBundle.getString(WRITE_KEY_METADATA)
                if (writeKey.isNullOrEmpty()) {
                    throw IllegalArgumentException("\n" +
                            "    ******************************************************************************\n" +
                            "    * The Aiactiv SDK was initialized incorrectly. Clients should follow the     *\n" +
                            "    * instructions here:                                                         *\n" +
                            "    * https://googlemobileadssdk.page.link/admob-android-update-manifest         *\n" +
                            "    * to add a valid Write Key inside the AndroidManifest.                       *\n" +
                            "    ******************************************************************************")
                }

                return Result.success(writeKey)
            }
        }

        private fun getAdNetworkWriteKey(): Result<String?> {
            return runCatching {
                val appBundle = getAppBundle()
                val writeKey = appBundle.getString(WRITE_KEY_ADNETWORK_METADATA)
                return Result.success(writeKey)
            }
        }

        fun build(): Configuration {
            val writeKey = getWriteKey().getOrThrow()
            val writeKeyForAdNetwork = getAdNetworkWriteKey().getOrNull()
            return Configuration(writeKey, writeKeyForAdNetwork)
        }
    }
}