package io.aiactiv.sdk

import android.app.Activity
import android.app.Application.ActivityLifecycleCallbacks
import android.content.pm.PackageInfo
import android.os.Build
import android.os.Bundle
import androidx.lifecycle.DefaultLifecycleObserver
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleObserver
import androidx.lifecycle.LifecycleOwner
import java.util.concurrent.atomic.AtomicBoolean
import java.util.concurrent.atomic.AtomicInteger

internal class AnalyticsActivityLifeCycleCallbacks(
    private val analytics: Analytics,
    private val packageInfo: PackageInfo,
    private val shouldTrackApplicationLifecycleEvents: Boolean,
    private val useNewLifecycleMethods: Boolean,
    private val trackDeepLinks: Boolean,
    private val shouldRecordScreenView: Boolean,
): ActivityLifecycleCallbacks, DefaultLifecycleObserver {

    private val trackedApplicationLifecycleEvents = AtomicBoolean(false)
    private val numberOfActivities = AtomicInteger(1)
    private val isChangingActivityConfigurations = AtomicBoolean(false)
    private val firstLaunch = AtomicBoolean(false)

    private val stubOwner = LifecycleOwner {
        return@LifecycleOwner object : Lifecycle() {
            override fun addObserver(observer: LifecycleObserver) {
                // No-op
            }

            override fun removeObserver(observer: LifecycleObserver) {
                // No-op
            }

            override fun getCurrentState(): State {
                return State.DESTROYED
            }
        }
    }

    override fun onStop(owner: LifecycleOwner) {
        if (shouldTrackApplicationLifecycleEvents &&
                numberOfActivities.decrementAndGet() == 0 &&
                isChangingActivityConfigurations.get().not()) {
            analytics.track("Application Backgrounded")
        }
    }

    override fun onStart(owner: LifecycleOwner) {
        if (shouldTrackApplicationLifecycleEvents &&
                numberOfActivities.incrementAndGet() == 1 &&
                isChangingActivityConfigurations.get().not()) {
            val properties = Properties()
            if (firstLaunch.get()) {
                properties.apply {
                    putValue("version", packageInfo.versionName)
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
                        putValue("build", packageInfo.longVersionCode.toString())
                    } else {
                        putValue("build", packageInfo.versionCode)
                    }
                }
            }
            properties.putValue("from_background", firstLaunch.getAndSet(false).not())
            analytics.track("Application Opened", properties)
        }
    }

    override fun onCreate(owner: LifecycleOwner) {
        if (trackedApplicationLifecycleEvents.getAndSet(true).not() &&
                shouldTrackApplicationLifecycleEvents) {
            numberOfActivities.set(0)
            firstLaunch.set(true)
            analytics.trackApplicationLifecycleEvents()
        }
    }

    override fun onActivityCreated(activity: Activity, savedInstanceState: Bundle?) {
        analytics.performRun(IntegrationOperation.onActivityCreated(activity, savedInstanceState))
        if (useNewLifecycleMethods.not()) {
            onCreate(stubOwner)
        }

        if (trackDeepLinks) {
            trackDeepLink(activity)
        }
    }

    override fun onActivityStarted(activity: Activity) {
        if (shouldRecordScreenView) {
            analytics.recordScreenView(activity)
        }
        analytics.performRun(IntegrationOperation.onActivityStarted(activity))
    }

    override fun onActivityResumed(activity: Activity) {
        analytics.performRun(IntegrationOperation.onActivityResumed(activity))
        if (useNewLifecycleMethods.not()) {
            onResume(stubOwner)
        }
    }

    override fun onActivityPaused(activity: Activity) {
        analytics.performRun(IntegrationOperation.onActivityPaused(activity))
        if (useNewLifecycleMethods.not()) {
            onPause(stubOwner)
        }
    }

    override fun onActivityStopped(activity: Activity) {
        analytics.performRun(IntegrationOperation.onActivityStopped(activity))
        if (useNewLifecycleMethods.not()) {
            onStop(stubOwner)
        }
    }

    override fun onActivitySaveInstanceState(activity: Activity, outState: Bundle) {
        analytics.performRun(IntegrationOperation.onActivitySaveInstanceState(activity, outState))
    }

    override fun onActivityDestroyed(activity: Activity) {
        analytics.performRun(IntegrationOperation.onActivityDestroyed(activity))
        if (useNewLifecycleMethods.not()) {
            onDestroy(stubOwner)
        }
    }

    private fun trackDeepLink(activity: Activity) {
        val intent = activity.intent
        if (intent == null || intent.data == null) {
            return
        }

        val properties = Properties()
        activity.referrer?.let { referer ->
            properties.putReferrer(referer.toString())
        }

        intent.data?.let { uri ->
            try {
                for (parameter in uri.queryParameterNames) {
                    val value = uri.getQueryParameter(parameter)
                    if (value.isNullOrEmpty().not()) {
                        properties.put(parameter, value)
                    }
                }
            } catch (e: Exception) {
                analytics.logger.error(e,"failed to get uri prams for $uri")
            }

            properties.put("url", uri.toString())
            analytics.track("Deep Link Opened", properties)
        }
    }

    class Builder {

        private val shouldTrackApplicationLifecycleEvents: Boolean = true
        private val useNewLifecycleMethods: Boolean = true
        private val trackDeepLinks: Boolean = true
        private val shouldRecordScreenView: Boolean = true

        fun build(analytics: Analytics, packageInfo: PackageInfo) = AnalyticsActivityLifeCycleCallbacks(
            analytics,
            packageInfo,
            shouldTrackApplicationLifecycleEvents,
            useNewLifecycleMethods,
            trackDeepLinks,
            shouldRecordScreenView
        )
    }
}