package io.aiactiv.sdk

import android.app.Activity
import android.app.Application
import android.app.Application.ActivityLifecycleCallbacks
import android.content.Context
import android.content.pm.PackageManager
import android.os.Build
import io.aiactiv.sdk.analytics.AiactivIntegration
import io.aiactiv.sdk.analytics.AnalyticsContext
import io.aiactiv.sdk.analytics.Traits
import io.aiactiv.sdk.analytics.integrations.*
import io.aiactiv.sdk.analytics.integrations.Integration
import io.aiactiv.sdk.internal.*
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import java.util.*

internal class Analytics(
    val context: Context,
    val writeKey: String,
    val tag: String,
    val logger: Logger,
    val cartographer: Cartographer,
    private val factories: List<Integration.Factory>,
    private val analyticsContext: AnalyticsContext,
    private val nanosecondTimestamps: Boolean,
    private val defaultOptions: Options,
    private val sourceMiddleware: MutableList<Middleware>,
    private val destinationMiddleware: Map<String, List<Middleware>>,
    private val traitsCache: Traits.Cache,
) {

    private var integrations = mutableMapOf<String, Integration<*>>()
    private val projectSettings: ProjectSettings

    val bundledIntegrations = mutableMapOf<String, Boolean>()

    private val activityLifeCycleCallbacks: ActivityLifecycleCallbacks

    init {
        projectSettings = getSettings()
        performInitializeIntegrations(projectSettings)

        activityLifeCycleCallbacks = AnalyticsActivityLifeCycleCallbacks.Builder().build(
            this,
            Utils.getPackageInfo(context))

        val application = context as Application
        application.registerActivityLifecycleCallbacks(activityLifeCycleCallbacks)
    }

    private fun getSettings(): ProjectSettings {
        val map: Map<String, Any?> = cartographer.fromJson(
            "{integrations={AiActiv={apiKey=" + this.writeKey + "}}, plan={track={__default={enabled=true, integrations={}}}, identify={__default={enabled=true}}, group={__default={enabled=true}}}, edgeFunction={}}"
        )
        return ProjectSettings.create(map as MutableMap<String, Any?>)
    }

    private fun performInitializeIntegrations(projectSettings: ProjectSettings) {
        if (projectSettings.isEmpty()) {
            throw AssertionError("ProjectSettings is empty!")
        }

        val integrationSettings = projectSettings.integrations()
        for (factory in factories) {
            if (integrationSettings.isNullOrEmpty()) {
                logger.debug("Integration Settings are empty")
                continue
            }

            val key = factory.key()
            if (key.isEmpty()) {
                throw AssertionError("The factory key is empty")
            }

            val settings = integrationSettings.getValueMap(key)
            if (settings.isNullOrEmpty()) {
                logger.debug("Integration $key is not enabled.")
            }

            val integration = factory.create(settings, this)
            integrations[key] = integration
            bundledIntegrations[key] = false
        }
    }

    fun trackApplicationLifecycleEvents() {
        val packageInfo = Utils.getPackageInfo(context)
        val versionCode = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
            packageInfo.longVersionCode
        } else {
            packageInfo.versionCode
        }
        val versionName = packageInfo.versionName

        val sharedPreferences = Utils.getAIActivSharedPreferences(context, tag)
        val previousVersionName = sharedPreferences.getString(VERSION_KEY, "null")
        val previousVersionCode = sharedPreferences.getInt(BUILD_KEY, -1)

        if (previousVersionCode == -1) {
            track(
                "Application Installed",
                Properties().apply {
                    putValue(VERSION_KEY, versionName)
                    putValue(BUILD_KEY, versionCode)
                }
            )
        } else if (versionCode != previousVersionCode) {
            track(
                "Application Updated",
                Properties().apply {
                    putValue(VERSION_KEY, versionName)
                    putValue(BUILD_KEY, versionCode.toString())
                    putValue("previous_$VERSION_KEY", previousVersionName!!)
                    putValue("previous_$BUILD_KEY", previousVersionCode.toString())
                }
            )
        }
    }

    /**
     * The track method is how you record any actions your users perform. Each action is known by a
     * name, like 'Purchased a T-Shirt'. You can also record properties specific to those actions.
     * For example a 'Purchased a Shirt' event might have properties like revenue or size.
     *
     * @param event Name of the event. Must not be null or empty.
     * @param properties {@link Properties} to add extra information to this call.
     * @param options To configure the call, these override the defaultOptions, to extend use
     *     #getDefaultOptions()
     * @throws IllegalArgumentException if event name is null or an empty string.
     */
    fun track(event: String, properties: Properties? = null, options: Options? = null) {
        if (event.isEmpty()) {
            throw IllegalArgumentException("event must not be empty.")
        }

        var finalProperties = Properties()
        if (properties != null) {
            finalProperties = properties
        }

        val timestamp = if (nanosecondTimestamps) NanoDate() else Date()
        val trackPayload = TrackPayload.Builder()
            .timestamp(timestamp)
            .event(event)
            .properties(finalProperties)
        fillAndEnqueue(trackPayload, options)
    }

    /**
     * Identify lets you tie one of your users and their actions to a recognizable [userId].
     * It also lets you record [Traits] about the user, like their email, name, account type,
     * etc.
     *
     * Traits and userId will be automatically cached and available on future sessions for the
     * same user. To update a trait on the server, call identify with the same user id (or null).
     * You can also use identify(traits: Traits)] for this purpose.
     *
     * @param userId Unique identifier which you recognize a user by in your own database. If this
     *     is null or empty, any previous id we have (could be the anonymous id) will be used.
     * @param newTraits Traits about the user.
     * @param options To configure the call, these override the defaultOptions, to extend use
     *     #getDefaultOptions()
     * @throws IllegalArgumentException if both [userId] and [newTraits] are not
     *     provided
     */
    fun identify(userId: String?, newTraits: Traits? = null, options: Options? = null) {
        if (userId.isNullOrEmpty() && newTraits.isNullOrEmpty()) {
            throw IllegalArgumentException("Either userId or some traits must be provided.")
        }

        val timestamp = if (nanosecondTimestamps) NanoDate() else Date()
        val traits = traitsCache.get()
        traits?.let {
            if (userId != null) {
                it.putUserId(userId)
            }
            if (newTraits != null) {
                it.putAll(newTraits)
            }

            traitsCache.set(it)
            analyticsContext.setTraits(it)
        }

        val builder = IdentifyPayload.Builder()
            .timestamp(timestamp)
            .traits(traitsCache.get()!!)

        fillAndEnqueue(builder, options)
    }

    /** @see identify(String, Traits, Options) */
    fun identify(traits: Traits) {
        identify(null, traits, null)
    }

    fun screen(category: String?, name: String?, properties: Properties? = null, options: Options? = null) {
        if (category.isNullOrEmpty() && name.isNullOrEmpty()) {
            throw IllegalArgumentException("either category or name must be provided.")
        }

        val timestamp = if (nanosecondTimestamps) NanoDate() else Date()
        val finalProperties = properties ?: Properties()

        val builder = ScreenPayload.Builder()
            .timestamp(timestamp)
            .name(name)
            .category(category)
            .properties(finalProperties)

        fillAndEnqueue(builder, options)
    }

    fun screen(name: String?, properties: Properties? = null, options: Options? = null) {
        screen(null, name, properties, options)
    }

    private fun enqueue(payload: BasePayload) {
        logger.verbose("Create payload $payload")
        val chain = MiddleChainRunner(0, payload, sourceMiddleware, object : Middleware.Callback {
            override fun invoke(payload: BasePayload) {
                run(payload)
            }
        })
        chain.proceed(payload)
    }

    private fun run(payload: BasePayload) {
        logger.verbose("Running payload $payload")
        val operation = IntegrationOperation.aiactivEvent(payload, destinationMiddleware)
        performRun(operation)
    }

    private fun fillAndEnqueue(builder: BasePayload.Builder<*, *>, options: Options?) {
        var finalOptions = defaultOptions
        if (options != null) {
            finalOptions = options
        }

        val contextCopy = AnalyticsContext(LinkedHashMap(analyticsContext.size))
        contextCopy.putAll(analyticsContext)
        contextCopy.putAll(finalOptions.context())

        builder.context(contextCopy)
        builder.anonymousId(contextCopy.traits()?.anonymousId())
        builder.integrations(finalOptions.integrations())
        builder.nanosecondTimestamps(nanosecondTimestamps)

        val cachedUserId = contextCopy.traits()?.userId()
        if (builder.isUserIdSet().not() && cachedUserId.isNullOrEmpty().not()) {
            builder.userId(cachedUserId)
        }

        enqueue(builder.build())
    }

    fun performRun(operation: IntegrationOperation) {
        integrations.entries.forEach { integration ->
            val key = integration.key
            val startTime = System.nanoTime()
            operation.run(key, integration.value, projectSettings)
            val endTime = System.nanoTime()
            // val duration = TimeUnit.NANOSECONDS.toMillis(endTime - startTime)
            logger.debug("Ran $operation on integration $key in ${endTime - startTime} ns.")
        }
    }

    fun recordScreenView(activity: Activity) {
        val packageManager = activity.packageManager
        try {
            val info = packageManager.getActivityInfo(activity.componentName, PackageManager.GET_META_DATA)
            val activityLabel = info.loadLabel(packageManager)
            screen(activityLabel.toString())
        } catch (e: PackageManager.NameNotFoundException) {
            throw AssertionError("Activity not found: $e")
        } catch (e: Exception) {
            logger.error(e, "Unable to track screen view for $activity")
        }
    }

    companion object {

        private const val VERSION_KEY = "version"
        private const val BUILD_KEY = "build"

        inline fun build(context: Context, writeKey: String, logger: Logger, block: Builder.() -> Unit): Analytics {
            return Builder(context, writeKey, logger).apply(block).build()
        }
    }

    class Builder(
        private val context: Context,
        private val writeKey: String,
        private val logger: Logger,
    ) {

        private val application: Application = context.applicationContext as Application
        private val tag: String = writeKey

        private var nanosecondTimeStamps = false
        private var collectDeviceID = Utils.DEFAULT_COLLECT_DEVICE_ID

        private var factories = mutableListOf<Integration.Factory>()

        private var defaultOptions = Options()
        private var sourceMiddleware: MutableList<Middleware>? = null
        private var destinationMiddleware: MutableMap<String, MutableList<Middleware>>? = null

        private val scope = CoroutineScope(Dispatchers.IO)

        fun build(): Analytics {

            val cartographer = Cartographer.INSTANCE
            val traitsCache = Traits.Cache(application, cartographer, tag)
            if (traitsCache.isSet() || traitsCache.get() == null) {
                traitsCache.set(Traits.create())
            }

            val analyticsContext = AnalyticsContext.create(application, traitsCache.get()!!, collectDeviceID)
            analyticsContext.attachAdvertisingId(context, logger, scope)

            factories.add(AiactivIntegration.FACTORY)

            val srcMiddleware = this.sourceMiddleware ?: mutableListOf()
            val destMiddleware = this.destinationMiddleware ?: emptyMap()

            return Analytics(
                context,
                writeKey,
                tag,
                logger,
                cartographer,
                factories,
                analyticsContext,
                nanosecondTimeStamps,
                defaultOptions,
                srcMiddleware,
                destMiddleware,
                traitsCache,
            )
        }

        fun nanosecondTimeStamps(): Builder {
            this.nanosecondTimeStamps = true
            return this
        }

        fun collectDeviceID(collect: Boolean): Builder {
            this.collectDeviceID = collect
            return this
        }

        fun defaultOptions(options: Options): Builder {
            this.defaultOptions = options
            this.defaultOptions.integrations().entries.forEach {
                if (it.value is Boolean) {
                    this.defaultOptions.setIntegration(it.key, it.value as Boolean)
                } else {
                    this.defaultOptions.setIntegration(it.key, true)
                }
            }
            return this
        }
        
//        fun useSourceMiddleware(middleware: Middleware): Builder {
//            if (sourceMiddleware == null) {
//                sourceMiddleware = mutableListOf()
//            }
//            if (sourceMiddleware!!.contains(middleware)) {
//                throw IllegalStateException("Source Middleware is already registered.")
//            }
//            sourceMiddleware!!.add(middleware)
//            return this
//        }
//
//        fun useDestinationMiddleware(key: String, middleware: Middleware): Builder {
//            if (destinationMiddleware == null) {
//                destinationMiddleware = mutableMapOf()
//            }
//            destinationMiddleware!!.let {
//                if (it[key] == null) {
//                    it[key] = mutableListOf()
//                }
//                if (it[key]!!.contains(middleware)) {
//                    throw IllegalStateException("Destination Middleware is already registered.")
//                }
//                it[key]!!.add(middleware)
//            }
//            return this
//        }
    }
}