package io.aiactiv.sdk

import android.Manifest
import android.content.Context
import io.aiactiv.sdk.internal.Logger
import io.aiactiv.sdk.internal.Utils

class AiactivSDK(private val context: Context) {

    private var logger: Logger = Logger("AiactivSDK", Logger.LogLevel.DEBUG)

    private lateinit var analytics: Analytics

    init {
        if (!Utils.hasPermission(context, Manifest.permission.INTERNET)) {
            throw IllegalArgumentException("INTERNET permission is required.")
        }
    }

    fun track(event: String) = this.analytics.track(event)

    private fun initAnalytics(writeKey: String) {
        this.analytics = Analytics.build(context, writeKey, logger) {
            nanosecondTimeStamps()
            collectDeviceID(true)
            defaultOptions(Options())
        }
    }

    private fun initAdNetwork(writeKey: String) {
        val adNetwork = AdNetwork.getInstance(context)
        adNetwork.initialize(writeKey)
    }

    companion object : SingletonHolder<AiactivSDK, Context>(::AiactivSDK) {

        internal fun setup(context: Context, configuration: Configuration): AiactivSDK {
            val instance = getInstance(context)

            val logger = instance.logger

            val writeKey = configuration.writeKey
            if (writeKey.isEmpty()) {
                throw IllegalStateException(
                    "Couldn't start AiactivSDK without writeKey." +
                            "Please make sure aiactiv_write_key is not empty in your string resource"
                )
            }

            logger.debug("Init Analytics module...")
            instance.initAnalytics(writeKey)

            val writeKeyForAdNetwork = configuration.writeKeyForAdNetwork
            if (writeKeyForAdNetwork != null) {
                logger.debug("Init AdNetwork module...")
                instance.initAdNetwork(writeKeyForAdNetwork)
            }

            return instance
        }

        /**
         * Return AiactivSDK single instance with context
         */
        fun with(context: Context): AiactivSDK {
            return getInstance(context)
        }

        fun track(context: Context, event: String) {
            with(context).track(event)
        }
    }
}