package io.aalam.common.router;

import java.util.ArrayList;

import io.aalam.common.*;

public class Mapper {
    ArrayList<Route> routeList;

    /**
    The constructor will be used only by the framework. The application
    using this library has no direct use by initializing a Mapper object.
    */
    public Mapper() {
        this.routeList = new ArrayList<Route>();
    }

    /**
    Returns a new sub mapper object. This can avoid adding handler to every
    mapper connect call.

    @param  handler Object of the handler class which has the action methods
    @return SubMapper Object to a submapper through which one can connect
                      urls with inputting the handler object on every
                      connect call.
    */
    public SubMapper subMapper(Object handler) {
        return new SubMapper(this, handler);
    }

    /**
    Connect a URL to the framework's url router.

    @param url The url path.
    @param methods Array of method names that apply on this url
    @param handler Object of the handler class.
    @param action The name of a public method defined in the handler
                  class that processes the functionality for this url.
    */
    public void connect(String url, String[] methods, Object handler,
                        String action) throws Exception {
        this.connect(url, methods, handler, action, null);
    }

    /**
    Connect a URL to the framework's url router.

    @param url The url path.
    @param methods Array of method names that apply on this url
    @param handler Object of the handler class.
    @param action The name of a public method defined in the handler
                  class that processes the functionality for this url.
    @param permissions Permissions object for this URL

    @see Permissions
    */
    public void connect(String url, String[] methods, Object handler,
                        String action, Permissions permissions) throws Exception {
        Route r = new Route(url, methods, handler, action, permissions);
        routeList.add(r);
    }

    private RouteMatch _match(HttpRequest req) {
        for (Route r : routeList) {
            String[] values = r.match(req);
            if (values != null) {
                return new RouteMatch(r, values);
            }
        }
        return (RouteMatch)null;
    }

    /**
    This method is used by the framework. The application has no direct use in
    calling this method.

    @param req The http request object
    */
    public RouteMatch match(HttpRequest req) {
        RouteMatch obj = this._match(req);
        if (obj != null)
            req.setRouteMatch(obj);

        return obj;
    }

    /**
    Submapper class for the mapper object.
    */
    public class SubMapper {
        Mapper mapper;
        Object handler;

        /**
        @param handler Object of the handler class for all the URL action
                       methods connecting to this submapper.
        */
        public SubMapper(Mapper mapper, Object handler) {
            this.mapper = mapper;
            this.handler = handler;
        }

        /**
        Connect a URL to the framework's url router, using handler that this
        submapper is initialized with.

        @param url The url path.
        @param methods Array of method names that apply on this url
        @param action The name of a public method defined in the handler
                      class that processes the functionality for this url.
        @param permissions Permissions object for this URL
        */
        public void connect(String url, String[] methods,
                            String action, Permissions permissions) throws Exception {
            this.mapper.connect(url, methods, this.handler, action, permissions);
        }

        /**
        Connect a URL to the framework's url router, using handler that this
        submapper is initialized with.

        @param url The url path.
        @param methods Array of method names that apply on this url
        @param action The name of a public method defined in the handler
                      class that processes the functionality for this url.
        */
        public void connect(String url, String[] methods,
                            String action) throws Exception {
            this.mapper.connect(url, methods, this.handler, action);
        }
    }
}
