package io.aalam.common;

import java.io.DataInputStream;
import java.io.InputStreamReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.security.InvalidKeyException;
import java.security.KeyFactory;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;

import io.netty.handler.codec.http.QueryStringEncoder;
import io.netty.handler.codec.http.HttpHeaders;

import org.apache.commons.codec.binary.Base64;

import org.bouncycastle.util.io.pem.PemObject;
import org.bouncycastle.util.io.pem.PemReader;

/**
Utility functions of this framework
*/
public class Utils {

    private static PrivateKey getPrivateKey(String keyPath) throws Exception {
        PemReader pemReader = new PemReader(new InputStreamReader(new FileInputStream(keyPath)));
        byte[] content = pemReader.readPemObject().getContent();
        PKCS8EncodedKeySpec privKeySpec = new PKCS8EncodedKeySpec(content);
        KeyFactory kf = KeyFactory.getInstance("RSA", "BC");
        return kf.generatePrivate(privKeySpec);
    }

    private static String getSignature(String message, String keyFile) {
        byte[] sig = null;
        try {
            PrivateKey pk = getPrivateKey(keyFile);
            MessageDigest md = MessageDigest.getInstance("SHA-1");
            md.update(message.getBytes());
            byte[] messageHash = md.digest();
            Signature s = Signature.getInstance("NonewithRSAandMGF1");
            s.initSign(pk);
            s.update(messageHash);
            sig = s.sign();
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        } catch (InvalidKeyException e) {
            e.printStackTrace();
        } catch (SignatureException e) {
            e.printStackTrace();
        } catch (Exception e) {
            e.printStackTrace();
        }

        return Base64.encodeBase64String(sig);
    }

    /**
    Send a signed request. Use requestLocalServer methods instead of this.

    @param prefix The prefix for the signature. This will be identified as
                  from() in the request.auth object of the receiver
    @param signatureKey The header value in which the signature will be put in
                        the response
    @param method The HTTP method
    @param path The URL path
    @param host   The host to send the request to. The host can verify the
                  signature only if this app's public key is present in it.
    @param params Key-Value map of url parmeters
    @param contentType Type of the content to be sent with the request
    @param data Data to be sent with the request
    @param scheme URL scheme like http, https, http+unix, etc
    */
    public static HttpResponse signAndSend(
                              String prefix, String signatureKey, String method,
                              String host, String path, Map<String, String> params,
                              String contentType, byte[] data, String scheme) {
        QueryStringEncoder enc = new QueryStringEncoder(path);
        String message = prefix + "#" + path;
        if (params != null) {
            message += "?";
            Iterator it = params.entrySet().iterator();
            while (it.hasNext()) {
                Map.Entry<String, String> entry = (Map.Entry<String, String>)it.next();
                enc.addParam(entry.getKey(), entry.getValue());
                message += (entry.getKey() + "=" + entry.getValue());
                if (it.hasNext())
                    message += "&";
            }
        }

        String signature = getSignature(message, Config.privKey);
        Map<String, String> headers = new HashMap<String, String>();
        headers.put(signatureKey, prefix + ";" + signature);
        if (contentType != null)
            headers.put(HttpHeaders.Names.CONTENT_TYPE, contentType);

        String uri = scheme + "://" + host + enc.toString();
        try {
            return HttpClient.request(method, uri, headers, params, data);
        } catch (Exception e) {
            return null;
        }
    }

    /**
    Send a signed request to other internal applications. If the application
    makes the request without using this method, those request will be seen
    as anonymous request by the receiving application.

    @param method The HTTP method
    @param url The URL path
    @param params Key-Value map of url parmeters
    @param contentType Type of the content to be sent with the request
    @param data Data to be sent with the request
    @param scheme URL scheme like http, https, http+unix, etc
    @param user The email id of the user on whose behalf this request is sent.
    */
    public static HttpResponse requestLocalServer(String method, String url,
                                                  Map<String, String> params,
                                                  String contentType, byte[] data,
                                                  String scheme, String user) {
        String prefix = Config.appProviderCode + "/" + Config.appCode;

        if (user != null) {
            prefix += "/" + user;
        }
        return signAndSend(prefix, "X-Auth-Internal", method,
                           "localhost", url, params, contentType, data, scheme);
    }

    /**
    Send a signed request to other internal applications. If the application
    makes the request without using this method, those request will be seen
    as anonymous request by the receiving application.

    @param method The HTTP method
    @param url The URL path
    @param params Key-Value map of url parmeters
    @param contentType Type of the content to be sent with the request
    @param data Data to be sent with the request
    */
    public static HttpResponse requestLocalServer(String method, String url,
                                                  Map<String, String> params,
                                                  String contentType, byte[] data) {
        return requestLocalServer(method, url, params, contentType, data, "http", null);
    }

    /**
    Send a signed request to other internal applications. If the application
    makes the request without using this method, those request will be seen
    as anonymous request by the receiving application.

    @param method The HTTP method
    @param url The URL path
    @param params Key-Value map of url parmeters
    */
    public static HttpResponse requestLocalServer(String method, String url,
                                                  Map<String, String> params) {
        return requestLocalServer(method, url, params, null, null, "http", null);
    }

}
